<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramCallbackData;
use App\Models\User;
use App\Models\DepositRequest;
use App\Services\TelegramService;
use App\Services\BitGoService;
use App\Services\QrService;
use App\Traits\HasBotTranslations;

class CreateDepositAction
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram,
        private BitGoService $bitgo,
        private QrService $qr
    ) {}

    public function execute(TelegramCallbackData $callback): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();

        if (!$user) {
            return;
        }

        $lang = $user->language ?? 'ru';
        $general = app(\App\Settings\GeneralSettings::class);
        $label = "user_{$user->id}_deposit_" . time();

        // Show typing indicator while processing
        $this->telegram->sendChatAction($callback->message->chat->id, 'typing');

        try {
            $addressData = $this->bitgo->createAddress($label);

            $depositRequest = DepositRequest::create([
                'user_id' => $user->id,
                'provider' => 'bitgo',
                'chain' => 'btc',
                'address' => $addressData['address'],
                'label' => $label,
                'status' => 'open',
            ]);

            // Generate QR code and save as temporary file
            $qrDataUri = $this->qr->generateBitcoinQr($addressData['address']);

            // Convert data URI to file
            $qrImageData = base64_decode(explode(',', $qrDataUri)[1]);
            $tempFile = storage_path('app/temp_qr_' . $depositRequest->id . '.png');
            file_put_contents($tempFile, $qrImageData);

            $text = __('bot.deposit_instructions_text', [
                'address' => "<code>{$addressData['address']}</code>",
                'min_deposit' => $general->min_deposit_usd,
                'support' => $general->support_username,
            ], $lang);

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '🔄 ' . __('bot.check_status', [], $lang), 'callback_data' => "depstatus:{$depositRequest->id}"],
                ],
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);

            // Send photo using file upload
            $this->telegram->sendPhotoFile(
                $callback->message->chat->id,
                $tempFile,
                $text,
                $keyboard
            );

            // Clean up temp file
            @unlink($tempFile);
        } catch (\Exception $e) {
            \Log::error('CreateDepositAction failed', [
                'error' => $e->getMessage(),
                'user_id' => $user->id
            ]);

            $text = __('bot.deposit_failed', [], $lang) . "\n\n";
            $text .= "Error: " . $e->getMessage() . "\n\n";
            $text .= __('bot.contact_support', ['support' => $general->support_username], $lang);

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => __('bot.try_again', [], $lang), 'callback_data' => 'dep:new'],
                ],
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                $text,
                $keyboard
            );
        }
    }
}
