<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramMessageData;
use App\Events\UserRegistered;
use App\Models\User;
use App\Models\Product;
use App\Services\ReferralService;
use App\Services\TelegramService;
use App\Traits\HasBotTranslations;

class ProcessStartCommand
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram,
        private ReferralService $referral
    ) {}

    public function execute(TelegramMessageData $message): void
    {
        // Check maintenance mode first
        $maintenance = app(\App\Settings\MaintenanceSettings::class);
        if ($maintenance->enabled) {
            $allowedUsers = array_filter(array_map('trim', explode(',', $maintenance->allowed_users)));
            $telegramId = (string)$message->from->id;

            if (!in_array($telegramId, $allowedUsers)) {
                $this->telegram->sendMessage($message->chat->id, $maintenance->message);
                return;
            }
        }

        $user = User::where('telegram_id', (string)$message->from->id)->first();
        $isNewUser = false;

        if (!$user) {
            $user = User::create([
                'telegram_id' => (string)$message->from->id,
                'username' => $message->from->username ?? null,
                'first_name' => $message->from->first_name ?? null,
                'last_name' => $message->from->last_name ?? null,
                'name' => $message->from->first_name ?? 'User',
                'balance_usd' => 0,
                'terms_accepted' => false,
            ]);
            $isNewUser = true;

            // Process referral code if present in /start command
            if (isset($message->text) && str_contains($message->text, 'ref_')) {
                preg_match('/ref_([A-Z0-9]+)/', $message->text, $matches);
                if (isset($matches[1])) {
                    $this->referral->processReferralCode($user, $matches[1]);
                }
            }
        }

        // Check if user has accepted terms
        if (!$user->terms_accepted) {
            $this->showTermsAndConditions($message->chat->id);
            return;
        }

        // User has accepted terms, show normal welcome
        $lang = $user->language ?? 'ru';
        $general = app(\App\Settings\GeneralSettings::class);

        if ($user->wasRecentlyCreated) {
            event(new UserRegistered($user));
            $text = __('bot.new_user_welcome', ['site_name' => $general->site_name], $lang);
        } else {
            $productCount = Product::where('sold', false)
                ->where('deleted', false)
                ->count();
            $text = __('bot.returning_user_welcome', [
                'count' => $productCount,
                'balance' => number_format((float)$user->balance_usd, 2),
            ], $lang);
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => __('bot.browse_products', [], $lang), 'callback_data' => 'browse'],
                ['text' => __('bot.deposit_btc', [], $lang), 'callback_data' => 'dep:new'],
            ],
            [
                ['text' => __('bot.my_balance', [], $lang), 'callback_data' => 'bal'],
                ['text' => __('bot.my_purchases', [], $lang), 'callback_data' => 'pur:0'],
            ],
            [
                ['text' => __('bot.referral', [], $lang), 'callback_data' => 'referral'],
                ['text' => __('bot.language', [], $lang), 'callback_data' => 'language'],
            ],
            [
                ['text' => __('bot.my_reports', [], $lang), 'callback_data' => 'myreports'],
                ['text' => __('bot.rules_terms', [], $lang), 'callback_data' => 'rules'],
            ],
            [
                ['text' => __('bot.help', [], $lang), 'callback_data' => 'help'],
            ],
        ]);

        $this->telegram->sendMessage($message->chat->id, $text, $keyboard);
    }

    private function showTermsAndConditions(int $chatId, string $lang = 'ru'): void
    {
        $general = app(\App\Settings\GeneralSettings::class);

        $text = __('bot.welcome_terms', ['site_name' => $general->site_name], $lang) . "\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        $text .= __('bot.rules_text', [], $lang);
        $text .= "\n\n━━━━━━━━━━━━━━━━━━━━\n\n";
        $text .= __('bot.accept_terms_question', [], $lang);

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => __('bot.accept_terms', [], $lang), 'callback_data' => 'terms:accept'],
                ['text' => __('bot.decline_terms', [], $lang), 'callback_data' => 'terms:decline'],
            ],
        ]);

        $this->telegram->sendMessage($chatId, $text, $keyboard);
    }
}
