<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramCallbackData;
use App\Events\ProductPurchased;
use App\Models\User;
use App\Models\Product;
use App\Models\Order;
use App\Services\TelegramService;
use App\Traits\HasBotTranslations;
use Illuminate\Support\Facades\DB;

class PurchaseProductAction
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram
    ) {}

    public function execute(TelegramCallbackData $callback, int $productId): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();

        if (!$user) {
            return;
        }

        $product = Product::find($productId);

        $lang = $user?->language ?? 'ru';
        
        if (!$product || $product->sold || $product->deleted) {
            $text = __('bot.product_not_available', [], $lang);
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                $text,
                $keyboard
            );
            return;
        }

        if ($user->balance_usd < $product->price) {
            $needed = $product->price - $user->balance_usd;
            $text = __('bot.insufficient_balance', [], $lang) . "\n\n";
            $text .= __('bot.need_more', ['amount' => number_format($needed, 2)], $lang) . "\n\n";
            $text .= __('bot.current_balance_label', ['balance' => $user->balance_usd], $lang) . "\n";
            $text .= __('bot.product_price_label', ['price' => $product->price], $lang);

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => __('bot.deposit', [], $lang), 'callback_data' => 'dep:new'],
                ],
                [
                    ['text' => __('bot.back_to_category', [], $lang), 'callback_data' => "cat:{$product->category_id}:0"],
                ],
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                $text,
                $keyboard
            );
            return;
        }

        DB::transaction(function () use ($user, $product, $callback) {
            $order = Order::create([
                'user_id' => $user->id,
                'total_usd' => $product->price,
                'status' => 'paid',
            ]);

            $order->items()->create([
                'product_id' => $product->id,
                'qty' => 1,
                'unit_price_usd' => $product->price,
                'line_total_usd' => $product->price,
                'snapshot' => $product->toArray(),
            ]);

            $user->decrement('balance_usd', $product->price);

            $product->update([
                'sold' => true,
                'sold_to' => $user->telegram_id,
                'purchased_time' => now(),
            ]);

            event(new ProductPurchased($order));

            $text = __('bot.purchase_successful', [], $lang) . "\n\n";
            $text .= __('bot.product_label', ['title' => $product->title], $lang) . "\n";
            $text .= __('bot.amount_label', ['amount' => $product->price], $lang) . "\n\n";
            
            if ($product->details) {
                $text .= __('bot.product_details_label', [], $lang) . "\n\n";
                $text .= $product->details . "\n\n";
            }

            $text .= __('bot.new_balance', ['balance' => $user->fresh()->balance_usd], $lang);

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => __('bot.my_purchases', [], $lang), 'callback_data' => 'pur:0'],
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                $text,
                $keyboard
            );
        });
    }
}
