<?php

namespace App\Actions\Telegram;

use App\Models\User;
use App\Models\Order;
use App\Models\ProductReport;
use App\Services\TelegramService;
use Illuminate\Support\Facades\Cache;

class ReportHandlers
{
    public function __construct(
        private TelegramService $telegram
    ) {}

    public function handleStartReport($callback, int $orderId): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();
        if (!$user) return;

        $order = Order::where('id', $orderId)
            ->where('user_id', $user->id)
            ->with('items.product')
            ->first();

        if (!$order) {
            $this->telegram->answerCallbackQuery($callback->id, "Order not found");
            return;
        }

        // Check if can report
        $general = app(\App\Settings\GeneralSettings::class);
        $maxDays = $general->max_report_days;

        if ($order->created_at->diffInDays(now()) > $maxDays) {
            $text = "❌ <b>Cannot Report</b>\n\n";
            $text .= "You can only report issues within {$maxDays} days of purchase.\n";
            $text .= "This order is too old to report.";

            $keyboard = $this->telegram->createInlineKeyboard([[
                ['text' => '🔙 Back', 'callback_data' => "vieworder:{$orderId}"],
            ]]);

            $this->telegram->answerCallbackQuery($callback->id);
            $this->telegram->sendMessage($callback->message->chat->id, $text, $keyboard);
            return;
        }

        // Check for existing pending report
        $existingReport = ProductReport::where('order_id', $orderId)
            ->whereIn('status', ['pending', 'under_review'])
            ->first();

        if ($existingReport) {
            $text = "⚠️ <b>Report Already Exists</b>\n\n";
            $text .= "You already have a pending report for this order.\n";
            $text .= "Ticket #{$existingReport->id}\n\n";
            $text .= "Check status in 📋 My Reports";

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '📋 View Report', 'callback_data' => "viewreport:{$existingReport->id}"],
                ],
                [
                    ['text' => '🔙 Back', 'callback_data' => "vieworder:{$orderId}"],
                ],
            ]);

            $this->telegram->answerCallbackQuery($callback->id);
            $this->telegram->sendMessage($callback->message->chat->id, $text, $keyboard);
            return;
        }

        // Start report process
        $text = "🚨 <b>Report Issue</b>\n\n";
        $text .= "Order #{$order->id}\n";
        $text .= "Total: \${$order->total_usd}\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        $text .= "Please describe what's wrong with the product:\n\n";
        $text .= "Example:\n";
        $text .= "• Login credentials don't work\n";
        $text .= "• Product link is broken\n";
        $text .= "• Wrong product delivered\n\n";
        $text .= "Type your issue below:";

        // Store state in cache
        Cache::put("report_state:{$user->id}", [
            'step' => 'awaiting_reason',
            'order_id' => $orderId,
        ], now()->addMinutes(30));

        $this->telegram->answerCallbackQuery($callback->id);
        $this->telegram->sendMessage($callback->message->chat->id, $text);
    }

    public function handleReportReason($message, User $user, string $reason): void
    {
        $state = Cache::get("report_state:{$user->id}");
        if (!$state || $state['step'] !== 'awaiting_reason') {
            return;
        }

        // Update state
        $state['reason'] = $reason;
        $state['step'] = 'awaiting_screenshot';
        Cache::put("report_state:{$user->id}", $state, now()->addMinutes(30));

        $lang = $user->language ?? 'ru';

        $text = "📷 <b>Screenshot Evidence (Optional)</b>\n\n";
        $text .= __('bot.report_instructions', [], $lang);

        $keyboard = $this->telegram->createInlineKeyboard([[
            ['text' => '⏭️ Skip Screenshot', 'callback_data' => 'reportskip'],
        ]]);

        $this->telegram->sendMessage($message->chat->id, $text, $keyboard);
    }



    public function handleReportScreenshot($message, User $user, string $screenshotUrl): void
    {
        $state = Cache::get("report_state:{$user->id}");
        if (!$state || $state['step'] !== 'awaiting_screenshot') {
            return;
        }

        // Validate URL
        $pattern = '/^(https?:\/\/)?(prnt\.sc|i\.imgur\.com)\/.+$/i';
        if (!preg_match($pattern, $screenshotUrl)) {
            $text = "❌ Invalid screenshot URL.\n\n";
            $text .= "Please use https://prnt.sc/ or paste a valid link.\n";
            $text .= "Or click Skip to continue without screenshot.";

            $keyboard = $this->telegram->createInlineKeyboard([[
                ['text' => '⏭️ Skip Screenshot', 'callback_data' => 'reportskip'],
            ]]);

            $this->telegram->sendMessage($message->chat->id, $text, $keyboard);
            return;
        }

        $this->createReport($message->chat->id, $user, $state, $screenshotUrl);
    }

    public function handleReportSkip($callback, User $user): void
    {
        $state = Cache::get("report_state:{$user->id}");
        if (!$state || $state['step'] !== 'awaiting_screenshot') {
            return;
        }

        $this->createReport($callback->message->chat->id, $user, $state, null);
        $this->telegram->answerCallbackQuery($callback->id);
    }

    private function createReport(int $chatId, User $user, array $state, ?string $screenshotUrl): void
    {
        $order = Order::with('items')->find($state['order_id']);
        if (!$order) return;

        // Get first order item (or could loop through all)
        $orderItem = $order->items->first();
        if (!$orderItem) return;

        // Create report
        $report = ProductReport::create([
            'user_id' => $user->id,
            'order_id' => $order->id,
            'order_item_id' => $orderItem->id,
            'product_id' => $orderItem->product_id,
            'reason' => $state['reason'],
            'screenshot_url' => $screenshotUrl,
            'status' => 'pending',
            'refund_amount' => $order->total_usd,
        ]);

        // Clear state
        Cache::forget("report_state:{$user->id}");

        $text = "✅ <b>Report Submitted!</b>\n\n";
        $text .= "Ticket #{$report->id}\n";
        $text .= "Order: #{$order->id}\n";
        $text .= "Amount: \${$order->total_usd}\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        $text .= "Your Issue:\n{$state['reason']}\n\n";
        if ($screenshotUrl) {
            $text .= "Screenshot: {$screenshotUrl}\n\n";
        }
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        $text .= "We'll review your report within 24 hours.\n";
        $text .= "Track status in 📋 My Reports";

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '📋 My Reports', 'callback_data' => 'myreports'],
            ],
            [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        $this->telegram->sendMessage($chatId, $text, $keyboard);
    }

    public function handleMyReports($callback): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();
        if (!$user) return;

        $reports = ProductReport::where('user_id', $user->id)
            ->with(['order', 'product'])
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        if ($reports->isEmpty()) {
            $text = "📋 <b>My Reports</b>\n\n";
            $text .= "You haven't submitted any reports yet.\n\n";
            $text .= "If you have issues with a product, you can report it from your order details.";

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '📦 My Purchases', 'callback_data' => 'pur:0'],
                ],
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);
        } else {
            $text = "📋 <b>My Reports</b>\n\n";
            $text .= "Click on a report to view details:\n\n";

            $buttons = [];
            foreach ($reports as $report) {
                $statusIcon = match($report->status) {
                    'pending' => '⏳',
                    'under_review' => '🔍',
                    'approved' => '✅',
                    'denied' => '❌',
                    'refunded' => '💰',
                    default => '📋',
                };

                $productName = $report->product ? $report->product->title : 'Product';
                $buttons[] = [
                    ['text' => "{$statusIcon} #{$report->id} - {$productName} - \${$report->refund_amount}", 'callback_data' => "viewreport:{$report->id}"],
                ];
            }

            $buttons[] = [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ];

            $keyboard = $this->telegram->createInlineKeyboard($buttons);
        }

        $this->telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }

    public function handleViewReport($callback, int $reportId): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();
        if (!$user) return;

        $report = ProductReport::where('id', $reportId)
            ->where('user_id', $user->id)
            ->with(['order', 'product', 'admin'])
            ->first();

        if (!$report) {
            $this->telegram->answerCallbackQuery($callback->id, "Report not found");
            return;
        }

        $text = "📋 <b>Report #{$report->id}</b>\n\n";
        $text .= "Status: {$report->status_badge}\n";
        $text .= "Order: #{$report->order_id}\n";
        $text .= "Product: " . ($report->product ? $report->product->title : 'N/A') . "\n";
        $text .= "Amount: \${$report->refund_amount}\n";
        $text .= "Reported: {$report->created_at->format('M d, Y H:i')}\n\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        $text .= "<b>Your Issue:</b>\n{$report->reason}\n\n";

        if ($report->screenshot_url) {
            $text .= "<b>Screenshot:</b>\n{$report->screenshot_url}\n\n";
        }

        if ($report->admin_response) {
            $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
            $text .= "<b>Admin Response:</b>\n{$report->admin_response}\n\n";
        }

        if ($report->status === 'refunded') {
            $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
            $text .= "✅ Refund of \${$report->refund_amount} has been credited to your balance.";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '🔙 Back to Reports', 'callback_data' => 'myreports'],
            ],
            [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        $this->telegram->answerCallbackQuery($callback->id);
        $this->telegram->sendMessage(
            $callback->message->chat->id,
            $text,
            $keyboard
        );
    }
}
