<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramCallbackData;
use App\Models\Category;
use App\Models\Product;
use App\Models\User;
use App\Services\TelegramService;
use App\Traits\HasBotTranslations;

class ShowCategoryProductsAction
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram
    ) {}

    public function execute(TelegramCallbackData $callback, int $categoryId, int $page = 0): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();
        $lang = $user?->language ?? 'ru';
        
        $category = Category::find($categoryId);

        if (!$category) {
            $this->telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                __('bot.category_not_found', [], $lang),
                $this->telegram->createInlineKeyboard([[
                    ['text' => __('bot.back', [], $lang), 'callback_data' => 'browse'],
                ]])
            );
            return;
        }

        // Check if this category has subcategories
        $subcategories = Category::where('parent_id', $categoryId)
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        // If has subcategories, show them instead of products
        if ($subcategories->isNotEmpty()) {
            $this->showSubcategories($callback, $category, $subcategories);
            return;
        }

        // Otherwise, show products in this category
        $perPage = 10;
        $totalProducts = Product::where('category_id', $categoryId)
            ->where('sold', false)
            ->where('deleted', false)
            ->count();

        $products = Product::where('category_id', $categoryId)
            ->where('sold', false)
            ->where('deleted', false)
            ->orderBy('created_at', 'desc')
            ->skip($page * $perPage)
            ->take($perPage)
            ->get();

        if ($products->isEmpty()) {
            $text = __('bot.category_products', ['category' => $category->name], $lang) . "\n\n";
            $text .= __('bot.no_products', [], $lang);

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => __('bot.back_to_categories', [], $lang), 'callback_data' => 'browse'],
                ],
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);
        } else {
            $text = __('bot.category_products', ['category' => $category->name], $lang) . "\n\n";
            $text .= __('bot.available_products', [], $lang) . "\n\n";

            $buttons = [];
            foreach ($products as $product) {
                $text .= "🔹 <b>{$product->title}</b>\n";
                $text .= "   " . __('bot.price', ['price' => $product->price], $lang) . "\n";
                if ($product->short_description) {
                    $text .= "   📝 " . substr($product->short_description, 0, 60);
                    if (strlen($product->short_description) > 60) {
                        $text .= "...";
                    }
                    $text .= "\n";
                }
                $text .= "\n";

                $buttons[] = [
                    ['text' => __('bot.buy_product', ['title' => $product->title, 'price' => $product->price], $lang), 'callback_data' => "buy:{$product->id}"],
                ];
            }

            // Pagination buttons
            $navButtons = [];
            $totalPages = (int) ceil($totalProducts / $perPage);
            $currentPage = $page + 1; // Convert to 1-based for display
            
            if ($page > 0) {
                $navButtons[] = ['text' => __('bot.previous', [], $lang), 'callback_data' => "cat:{$categoryId}:" . ($page - 1)];
            }
            
            // Only show Next button if we're not on the last page
            if ($currentPage < $totalPages) {
                $navButtons[] = ['text' => __('bot.next', [], $lang), 'callback_data' => "cat:{$categoryId}:" . ($page + 1)];
            }
            
            if (!empty($navButtons)) {
                $buttons[] = $navButtons;
            }

            $buttons[] = [
                ['text' => __('bot.back_to_categories', [], $lang), 'callback_data' => 'browse'],
            ];
            $buttons[] = [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ];

            $keyboard = $this->telegram->createInlineKeyboard($buttons);
        }

        $this->telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }

    private function showSubcategories(TelegramCallbackData $callback, Category $category, $subcategories): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();
        $lang = $user?->language ?? 'ru';
        
        $text = __('bot.category_products', ['category' => $category->name], $lang) . "\n\n";
        $text .= __('bot.select_subcategory', [], $lang) . "\n";

        $buttons = [];
        foreach ($subcategories as $subcategory) {
            $buttons[] = [
                ['text' => $subcategory->name, 'callback_data' => "cat:{$subcategory->id}:0"],
            ];
        }

        // Add back button
        $buttons[] = [
            ['text' => __('bot.back_to_categories', [], $lang), 'callback_data' => 'browse'],
        ];
        $buttons[] = [
            ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
        ];

        $keyboard = $this->telegram->createInlineKeyboard($buttons);

        $this->telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }
}
