<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramCallbackData;
use App\Models\User;
use App\Models\Product;
use App\Services\TelegramService;
use App\Traits\HasBotTranslations;

class ShowMainMenuAction
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram
    ) {}

    public function execute(TelegramCallbackData $callback): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();

        if (!$user) {
            return;
        }

        $lang = $user->language ?? 'ru';
        $productCount = Product::where('sold', false)
            ->where('deleted', false)
            ->count();

        $text = __('bot.returning_user_welcome', [
            'count' => $productCount,
            'balance' => number_format((float)$user->balance_usd, 2),
        ], $lang);

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => __('bot.browse_products', [], $lang), 'callback_data' => 'browse'],
                ['text' => __('bot.deposit_btc', [], $lang), 'callback_data' => 'dep:new'],
            ],
            [
                ['text' => __('bot.my_balance', [], $lang), 'callback_data' => 'bal'],
                ['text' => __('bot.my_purchases', [], $lang), 'callback_data' => 'pur:0'],
            ],
            [
                ['text' => __('bot.referral', [], $lang), 'callback_data' => 'referral'],
                ['text' => __('bot.language', [], $lang), 'callback_data' => 'language'],
            ],
            [
                ['text' => __('bot.my_reports', [], $lang), 'callback_data' => 'myreports'],
                ['text' => __('bot.rules_terms', [], $lang), 'callback_data' => 'rules'],
            ],
            [
                ['text' => __('bot.help', [], $lang), 'callback_data' => 'help'],
            ],
        ]);

        // Answer the callback query first
        $this->telegram->answerCallbackQuery($callback->id);
        
        // Try to edit the message
        $result = $this->telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
        
        // If edit failed (e.g., message is a photo), send a new message
        if (!($result['ok'] ?? false)) {
            $this->telegram->sendMessage(
                $callback->message->chat->id,
                $text,
                $keyboard
            );
        }
    }
}
