<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramCallbackData;
use App\Models\Product;
use App\Models\User;
use App\Services\TelegramService;
use App\Traits\HasBotTranslations;

class ShowProductDetailsAction
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram
    ) {}

    public function execute(TelegramCallbackData $callback, int $productId): void
    {
        $user = User::where('telegram_id', (string)$callback->from->id)->first();
        $lang = $user?->language ?? 'ru';
        
        $product = Product::find($productId);

        if (!$product || $product->sold || $product->deleted) {
            $this->telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                __('bot.product_not_available', [], $lang),
                $this->telegram->createInlineKeyboard([[
                    ['text' => __('bot.back', [], $lang), 'callback_data' => 'browse'],
                ]])
            );
            return;
        }

        $text = "🔹 <b>{$product->title}</b>\n\n";
        $text .= __('bot.product_price', ['price' => $product->price], $lang) . "\n\n";
        
        if ($product->short_description) {
            $text .= __('bot.product_description', [], $lang) . "\n{$product->short_description}\n\n";
        }
        
        if ($product->details) {
            $text .= __('bot.product_details', [], $lang) . "\n{$product->details}\n\n";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => __('bot.buy_now', [], $lang), 'callback_data' => "buy:{$product->id}"],
            ],
            [
                ['text' => __('bot.back_to_category', [], $lang), 'callback_data' => "cat:{$product->category_id}:0"],
            ],
            [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ],
        ]);

        $this->telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }
}
