<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramCallbackData;
use App\Models\User;
use App\Services\ReferralService;
use App\Services\TelegramService;
use App\Traits\HasBotTranslations;

class ShowReferralInfoAction
{
    use HasBotTranslations;

    public function __construct(
        private TelegramService $telegram,
        private ReferralService $referral
    ) {}

    public function execute(TelegramCallbackData $callback): void
    {
        $user = User::where('telegram_id', $callback->from->id)->first();

        if (!$user) {
            return;
        }

        $lang = $user->language ?? 'ru';
        $stats = $this->referral->getReferralStats($user);
        $referralLink = $this->referral->generateReferralLink($user);

        $text = __('bot.referral_title', [], $lang) . "\n\n";
        $text .= __('bot.your_referral_code', ['code' => $user->referral_code], $lang) . "\n\n";
        $text .= __('bot.referral_link', [], $lang) . "\n";
        $text .= "<code>{$referralLink}</code>\n\n";
        $text .= __('bot.referral_earnings', ['earnings' => number_format($stats['total_earnings'], 2)], $lang) . "\n";
        $text .= __('bot.total_referrals', ['count' => $stats['total_referrals']], $lang) . "\n";
        $text .= __('bot.active_referrals', ['count' => $stats['active_referrals']], $lang) . "\n";
        $text .= __('bot.referral_commission', [], $lang) . "\n\n";
        $text .= "📝 " . __('bot.referral_instructions', [], $lang);

        $buttons = [
            [
                ['text' => __('bot.share_link', [], $lang), 'url' => "https://t.me/share/url?url=" . urlencode($referralLink)],
            ],
            [
                ['text' => __('bot.back', [], $lang), 'callback_data' => 'menu'],
            ],
        ];

        $keyboard = $this->telegram->createInlineKeyboard($buttons);

        $this->telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }
}
