<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Category;
use App\Models\Product;
use App\Models\DepositRequest;
use App\Models\Deposit;
use App\Models\Order;

class CheckAppStatus extends Command
{
    protected $signature = 'app:status';
    protected $description = 'Check ToolStore application status and configuration';

    public function handle()
    {
        $this->info('🏪 ToolStore Application Status');
        $this->newLine();

        $this->checkEnvironment();
        $this->newLine();
        
        $this->checkDatabase();
        $this->newLine();
        
        $this->checkServices();
        $this->newLine();
        
        $this->showSummary();
        
        return 0;
    }

    protected function checkEnvironment()
    {
        $this->info('📋 Environment Configuration:');
        
        $checks = [
            'APP_URL' => config('app.url'),
            'TELEGRAM_BOT_TOKEN' => config('services.telegram.bot_token') ? '✅ Configured' : '❌ Missing',
            'TELEGRAM_WEBHOOK_SECRET' => config('services.telegram.webhook_secret') ? '✅ Configured' : '⚠️  Not set',
            'BITGO_API_KEY' => config('services.bitgo.api_key') ? '✅ Configured' : '❌ Missing',
            'BITGO_WALLET_ID' => config('services.bitgo.wallet_id') ? '✅ Configured' : '❌ Missing',
            'BITGO_TESTNET' => config('services.bitgo.testnet') ? 'true (testnet)' : 'false (production)',
            'QUEUE_CONNECTION' => config('queue.default'),
            'CACHE_DRIVER' => config('cache.default'),
        ];

        foreach ($checks as $key => $value) {
            $this->line("  {$key}: {$value}");
        }
    }

    protected function checkDatabase()
    {
        $this->info('💾 Database Status:');
        
        try {
            $users = User::count();
            $categories = Category::count();
            $products = Product::count();
            $availableProducts = Product::where('sold', false)->where('deleted', false)->count();
            $depositRequests = DepositRequest::count();
            $deposits = Deposit::where('status', 'confirmed')->count();
            $orders = Order::count();
            
            $this->line("  Users: {$users}");
            $this->line("  Categories: {$categories}");
            $this->line("  Products (total): {$products}");
            $this->line("  Products (available): {$availableProducts}");
            $this->line("  Deposit Requests: {$depositRequests}");
            $this->line("  Confirmed Deposits: {$deposits}");
            $this->line("  Orders: {$orders}");
            
            $totalBalance = User::sum('balance_usd');
            $this->line("  Total User Balance: \${$totalBalance}");
            
        } catch (\Exception $e) {
            $this->error('  ❌ Database connection failed: ' . $e->getMessage());
        }
    }

    protected function checkServices()
    {
        $this->info('🔧 Services Status:');
        
        $services = [
            'BitGoService' => class_exists('App\Services\BitGoService'),
            'RateService' => class_exists('App\Services\RateService'),
            'QrService' => class_exists('App\Services\QrService'),
            'TelegramService' => class_exists('App\Services\TelegramService'),
        ];
        
        foreach ($services as $service => $exists) {
            $status = $exists ? '✅' : '❌';
            $this->line("  {$status} {$service}");
        }
    }

    protected function showSummary()
    {
        $this->info('📝 Quick Commands:');
        $this->line('  php artisan telegram:test          - Test Telegram bot');
        $this->line('  php artisan telegram:set-webhook   - Set webhook');
        $this->line('  php artisan migrate                - Run migrations');
        $this->line('  php artisan db:seed                - Seed database');
        $this->line('  php artisan serve                  - Start server');
        $this->newLine();
        $this->info('📚 Documentation:');
        $this->line('  TELEGRAM-TESTING.md  - Testing guide');
        $this->line('  SETUP.md             - Setup instructions');
        $this->line('  README.md            - Project overview');
    }
}
