<?php

namespace App\Console\Commands;

use App\Models\Category;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ImportCategories extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'categories:import 
                            {database : The name of the source database}
                            {--host=127.0.0.1 : Database host}
                            {--port=3306 : Database port}
                            {--username=root : Database username}
                            {--password= : Database password}
                            {--clear : Clear existing categories before import}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import categories from old database structure (parent_category and category tables)';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $database = $this->argument('database');
        $host = $this->option('host');
        $port = $this->option('port');
        $username = $this->option('username');
        $password = $this->option('password');
        $clearExisting = $this->option('clear');

        // Configure temporary database connection
        config([
            'database.connections.import_source' => [
                'driver' => 'mysql',
                'host' => $host,
                'port' => $port,
                'database' => $database,
                'username' => $username,
                'password' => $password,
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
            ]
        ]);

        try {
            // Test connection
            DB::connection('import_source')->getPdo();
            $this->info("✓ Connected to source database: {$database}");
        } catch (\Exception $e) {
            $this->error("✗ Failed to connect to source database: " . $e->getMessage());
            return 1;
        }

        // Clear existing categories if requested
        if ($clearExisting) {
            if ($this->confirm('Are you sure you want to delete all existing categories?', false)) {
                Category::query()->delete();
                $this->info("✓ Cleared existing categories");
            } else {
                $this->info("Import cancelled");
                return 0;
            }
        }

        $this->info("\n📦 Starting category import...\n");

        // Import parent categories first
        $parentCategoriesImported = $this->importParentCategories();
        
        // Import subcategories
        $subCategoriesImported = $this->importSubCategories();

        $this->info("\n✅ Import completed successfully!");
        $this->info("   Parent categories imported: {$parentCategoriesImported}");
        $this->info("   Subcategories imported: {$subCategoriesImported}");
        $this->info("   Total categories: " . ($parentCategoriesImported + $subCategoriesImported));

        return 0;
    }

    /**
     * Import parent categories
     */
    protected function importParentCategories(): int
    {
        $this->info("Importing parent categories...");

        $parentCategories = DB::connection('import_source')
            ->table('parent_category')
            ->where('deleted', 0)
            ->get();

        $imported = 0;
        $bar = $this->output->createProgressBar($parentCategories->count());
        $bar->start();

        foreach ($parentCategories as $oldCategory) {
            try {
                Category::create([
                    'name' => $oldCategory->name,
                    'slug' => $this->generateUniqueSlug($oldCategory->cat_slug ?: $oldCategory->name),
                    'description' => null,
                    'parent_id' => null, // Parent categories have no parent
                    'is_active' => true,
                    'sort_order' => $oldCategory->id,
                ]);
                $imported++;
            } catch (\Exception $e) {
                $this->newLine();
                $this->warn("Failed to import parent category '{$oldCategory->name}': " . $e->getMessage());
            }
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        return $imported;
    }

    /**
     * Import subcategories
     */
    protected function importSubCategories(): int
    {
        $this->info("Importing subcategories...");

        $subCategories = DB::connection('import_source')
            ->table('category')
            ->where('deleted', 0)
            ->get();

        $imported = 0;
        $skipped = 0;
        $bar = $this->output->createProgressBar($subCategories->count());
        $bar->start();

        foreach ($subCategories as $oldCategory) {
            try {
                // Find the parent category by matching the parent_id from old database
                $parentCategory = null;
                
                if (!empty($oldCategory->parent_id)) {
                    // Try to find parent by old parent_category id
                    $oldParent = DB::connection('import_source')
                        ->table('parent_category')
                        ->where('id', $oldCategory->parent_id)
                        ->where('deleted', 0)
                        ->first();
                    
                    if ($oldParent) {
                        // Find the corresponding new parent category by name
                        $parentCategory = Category::where('name', $oldParent->name)
                            ->whereNull('parent_id')
                            ->first();
                    }
                }

                if (!$parentCategory) {
                    $skipped++;
                    $bar->advance();
                    continue;
                }

                Category::create([
                    'name' => $oldCategory->name,
                    'slug' => $this->generateUniqueSlug($oldCategory->cat_slug ?: $oldCategory->name),
                    'description' => $oldCategory->comments,
                    'parent_id' => $parentCategory->id,
                    'is_active' => true,
                    'sort_order' => $oldCategory->id,
                ]);
                $imported++;
            } catch (\Exception $e) {
                $this->newLine();
                $this->warn("Failed to import subcategory '{$oldCategory->name}': " . $e->getMessage());
            }
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        if ($skipped > 0) {
            $this->warn("⚠ Skipped {$skipped} subcategories (parent not found)");
        }

        return $imported;
    }

    /**
     * Generate a unique slug
     */
    protected function generateUniqueSlug(string $text): string
    {
        $slug = Str::slug($text);
        $originalSlug = $slug;
        $counter = 1;

        while (Category::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }
}
