<?php

namespace App\Console\Commands;

use App\Models\Category;
use App\Models\Product;
use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ImportProducts extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'products:import
                            {database : The name of the source database}
                            {--host=127.0.0.1 : Database host}
                            {--port=3306 : Database port}
                            {--username=root : Database username}
                            {--password= : Database password}
                            {--clear : Clear existing products before import}
                            {--unsell : Set all products as unsold (sold_to = null)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import products from old database structure';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $database = $this->argument('database');
        $host = $this->option('host');
        $port = $this->option('port');
        $username = $this->option('username');
        $password = $this->option('password');
        $clearExisting = $this->option('clear');
        $unsellAll = $this->option('unsell');

        // Configure temporary database connection
        config([
            'database.connections.import_source' => [
                'driver' => 'mysql',
                'host' => $host,
                'port' => $port,
                'database' => $database,
                'username' => $username,
                'password' => $password,
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
            ]
        ]);

        try {
            // Test connection
            DB::connection('import_source')->getPdo();
            $this->info("✓ Connected to source database: {$database}");
        } catch (\Exception $e) {
            $this->error("✗ Failed to connect to source database: " . $e->getMessage());
            return 1;
        }

        // Clear existing products if requested
        if ($clearExisting) {
            if ($this->confirm('Are you sure you want to delete all existing products?', false)) {
                Product::query()->delete();
                $this->info("✓ Cleared existing products");
            } else {
                $this->info("Import cancelled");
                return 0;
            }
        }

        $this->info("\n📦 Starting product import...\n");

        // Build category mapping
        $categoryMapping = $this->buildCategoryMapping();

        if (empty($categoryMapping)) {
            $this->error("✗ No categories found. Please import categories first.");
            return 1;
        }

        $this->info("✓ Built category mapping for " . count($categoryMapping) . " categories\n");

        // Import products
        $productsImported = $this->importProducts($categoryMapping, $unsellAll);

        $this->info("\n✅ Import completed successfully!");
        $this->info("   Products imported: {$productsImported}");

        return 0;
    }

    /**
     * Build category mapping from old category IDs to new category IDs
     */
    protected function buildCategoryMapping(): array
    {
        $mapping = [];

        // Get all categories from source database
        $oldCategories = DB::connection('import_source')
            ->table('category')
            ->where('deleted', 0)
            ->get();

        foreach ($oldCategories as $oldCategory) {
            // Find matching category in new database by name
            $newCategory = Category::where('name', $oldCategory->name)->first();

            if ($newCategory) {
                // Map old category ID to new category ID
                $mapping[$oldCategory->id] = $newCategory->id;
            }
        }

        return $mapping;
    }

    /**
     * Import products
     */
    protected function importProducts(array $categoryMapping, bool $unsellAll): int
    {
        $this->info("Importing products...");

        $products = DB::connection('import_source')
            ->table('products')
            ->where('deleted', 0)
            ->get();

        $imported = 0;
        $skipped = 0;
        $bar = $this->output->createProgressBar($products->count());
        $bar->start();

        foreach ($products as $oldProduct) {
            try {
                // Map category ID
                $categoryId = null;
                if (!empty($oldProduct->category)) {
                    $categoryId = $categoryMapping[$oldProduct->category] ?? null;
                }

                // Skip if category not found
                if (!$categoryId) {
                    $skipped++;
                    $bar->advance();
                    continue;
                }

                // Find seller (if exists in new database)
                if (!empty($oldProduct->seller_id)) {
                    $seller = User::find($oldProduct->seller_id);
                    $sellerId = $seller ? $seller->id : null;
                }

                // Parse price
                $price = is_numeric($oldProduct->prod_price) 
                    ? (float)$oldProduct->prod_price 
                    : 0.00;

                // Create product with reset values
                Product::create([
                    'seller_id' => $sellerId,
                    'category_id' => $categoryId,
                    'title' => $oldProduct->title ?: 'Untitled Product',
                    'short_description' => $oldProduct->meta_value ?: null,
                    'balance' => $oldProduct->balance,
                    'price' => $price,
                    'details' => $oldProduct->details,
                    'sold' => false,
                    'sold_to' => '',
                    'deleted' => false,
                    'refund' => false,
                    'purchased_time' => null,
                    'updated' => $oldProduct->updated,
                ]);

                $imported++;
            } catch (\Exception $e) {
                $this->newLine();
                $this->warn("Failed to import product ID {$oldProduct->id}: " . $e->getMessage());
            }
            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        if ($skipped > 0) {
            $this->warn("⚠ Skipped {$skipped} products (category not found)");
        }

        return $imported;
    }
}
