<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\TelegramService;
use Illuminate\Support\Facades\Http;

class TestTelegramBot extends Command
{
    protected $signature = 'telegram:test';
    protected $description = 'Test Telegram bot connection and configuration';

    public function handle()
    {
        $this->info('🤖 Testing Telegram Bot Configuration...');
        $this->newLine();

        $token = config('services.telegram.bot_token');
        $webhookSecret = config('services.telegram.webhook_secret');
        $appUrl = config('app.url');

        if (!$token) {
            $this->error('❌ TELEGRAM_BOT_TOKEN not configured');
            return 1;
        }

        $this->info('✅ Bot token: Configured');
        
        if ($webhookSecret) {
            $this->info('✅ Webhook secret: Configured');
        } else {
            $this->warn('⚠️  Webhook secret: Not configured (optional but recommended)');
        }

        $this->info('📍 App URL: ' . $appUrl);
        $this->newLine();

        $this->info('Fetching bot info...');
        
        try {
            $response = Http::get("https://api.telegram.org/bot{$token}/getMe");
            $data = $response->json();

            if ($data['ok'] ?? false) {
                $bot = $data['result'];
                $this->info('✅ Bot is accessible!');
                $this->info('Bot name: ' . $bot['first_name']);
                $this->info('Bot username: @' . $bot['username']);
                $this->info('Bot ID: ' . $bot['id']);
            } else {
                $this->error('❌ Failed to get bot info');
                return 1;
            }
        } catch (\Exception $e) {
            $this->error('❌ Connection failed: ' . $e->getMessage());
            return 1;
        }

        $this->newLine();
        $this->info('Checking webhook status...');

        try {
            $response = Http::get("https://api.telegram.org/bot{$token}/getWebhookInfo");
            $data = $response->json();

            if ($data['ok'] ?? false) {
                $webhook = $data['result'];
                
                if ($webhook['url']) {
                    $this->info('✅ Webhook is set');
                    $this->info('URL: ' . $webhook['url']);
                    $this->info('Pending updates: ' . ($webhook['pending_update_count'] ?? 0));
                    
                    if (isset($webhook['last_error_date'])) {
                        $this->warn('⚠️  Last error: ' . $webhook['last_error_message']);
                    }
                } else {
                    $this->warn('⚠️  Webhook not set');
                    $this->info('Run: php artisan telegram:set-webhook');
                }
            }
        } catch (\Exception $e) {
            $this->error('❌ Failed to check webhook: ' . $e->getMessage());
        }

        $this->newLine();
        $this->info('✅ Test complete!');
        $this->newLine();
        $this->info('Next steps:');
        $this->info('1. Set webhook: php artisan telegram:set-webhook');
        $this->info('2. Send /start to your bot');
        $this->info('3. Check logs: tail -f storage/logs/laravel.log');

        return 0;
    }
}
