<?php

namespace App\Filament\Clusters\Settings\Pages;

use App\Filament\Clusters\Settings\SettingsCluster;
use App\Jobs\BroadcastCustomMessage;
use App\Models\User;
use BackedEnum;
use Filament\Actions\Action;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;

class BroadcastMessage extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedMegaphone;

    protected static ?string $cluster = SettingsCluster::class;

    protected static ?string $navigationLabel = 'Broadcast Message';

    protected static ?string $title = 'Broadcast Custom Message';

    protected string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill([
            'message' => '',
        ]);
    }

    public function form(Schema $schema): Schema
    {
        $activeUsersCount = User::whereNotNull('telegram_id')
            ->count();

        return $schema
            ->statePath('data')
            ->schema([
                Section::make('Broadcast Custom Message')
                    ->description("Send a custom message to all active users ({$activeUsersCount} users)")
                    ->schema([
                        Textarea::make('message')
                            ->label('Message')
                            ->required()
                            ->rows(8)
                            ->placeholder('Enter your message here...')
                            ->helperText('You can use HTML formatting: <b>bold</b>, <i>italic</i>, <code>code</code>, <a href="url">link</a>')
                            ->columnSpanFull(),
                    ])
                    ->icon('heroicon-o-megaphone'),
            ]);
    }

    public function save(): void
    {
        $data = $this->form->getState();

        if (empty(trim($data['message']))) {
            Notification::make()
                ->danger()
                ->title('Message Required')
                ->body('Please enter a message to broadcast.')
                ->send();
            return;
        }

        // Get count of active users
        $activeUsersCount = User::whereNotNull('telegram_id')
            ->count();

        if ($activeUsersCount === 0) {
            Notification::make()
                ->warning()
                ->title('No Active Users')
                ->body('There are no active users to send the message to.')
                ->send();
            return;
        }

        // Dispatch the broadcast job
        BroadcastCustomMessage::dispatch($data['message']);

        Notification::make()
            ->success()
            ->title('Broadcast Queued')
            ->body("Your message is being sent to {$activeUsersCount} users.")
            ->send();

        // Clear the form
        $this->form->fill([
            'message' => '',
        ]);
    }

    protected function getFormActions(): array
    {
        $activeUsersCount = User::whereNotNull('telegram_id')
            ->count();

        return [
            Action::make('save')
                ->label("Send to {$activeUsersCount} Users")
                ->icon('heroicon-o-paper-airplane')
                ->color('success')
                ->requiresConfirmation()
                ->modalHeading('Confirm Broadcast')
                ->modalDescription("Are you sure you want to send this message to {$activeUsersCount} users?")
                ->modalSubmitActionLabel('Yes, Send Message')
                ->submit('save'),
        ];
    }
}
