<?php

namespace App\Filament\Clusters\Settings\Pages;

use App\Filament\Clusters\Settings\SettingsCluster;
use App\Models\Category;
use BackedEnum;
use Filament\Actions\Action;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class ImportCategories extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedArrowDownTray;

    protected static ?string $cluster = SettingsCluster::class;

    protected static ?string $navigationLabel = 'Import Categories';

    protected static ?string $title = 'Import Categories from Old Database';

    protected string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill([
            'database' => '',
            'host' => '127.0.0.1',
            'port' => '3306',
            'username' => 'root',
            'password' => '',
            'clear_existing' => false,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        $currentCategoriesCount = Category::count();

        return $schema
            ->statePath('data')
            ->schema([
                Section::make('Database Connection')
                    ->description('Enter the connection details for the source database')
                    ->schema([
                        TextInput::make('database')
                            ->label('Database Name')
                            ->required()
                            ->placeholder('old_database_name')
                            ->helperText('The name of the database containing parent_category and category tables'),
                        TextInput::make('host')
                            ->label('Host')
                            ->required()
                            ->default('127.0.0.1'),
                        TextInput::make('port')
                            ->label('Port')
                            ->required()
                            ->numeric()
                            ->default('3306'),
                        TextInput::make('username')
                            ->label('Username')
                            ->required()
                            ->default('root'),
                        TextInput::make('password')
                            ->label('Password')
                            ->password()
                            ->revealable(),
                    ])
                    ->columns(2),

                Section::make('Import Options')
                    ->description("Current categories in database: {$currentCategoriesCount}")
                    ->schema([
                        Checkbox::make('clear_existing')
                            ->label('Clear existing categories before import')
                            ->helperText('⚠️ Warning: This will delete all existing categories and their relationships')
                            ->default(false),
                    ]),
            ]);
    }

    public function import(): void
    {
        $data = $this->form->getState();

        // Test connection first
        config([
            'database.connections.test_import' => [
                'driver' => 'mysql',
                'host' => $data['host'],
                'port' => $data['port'],
                'database' => $data['database'],
                'username' => $data['username'],
                'password' => $data['password'],
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
            ]
        ]);

        try {
            DB::connection('test_import')->getPdo();
        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Connection Failed')
                ->body('Could not connect to the database: ' . $e->getMessage())
                ->send();
            return;
        }

        // Check if required tables exist
        try {
            $hasParentCategory = DB::connection('test_import')
                ->getSchemaBuilder()
                ->hasTable('parent_category');
            $hasCategory = DB::connection('test_import')
                ->getSchemaBuilder()
                ->hasTable('category');

            if (!$hasParentCategory || !$hasCategory) {
                Notification::make()
                    ->danger()
                    ->title('Missing Tables')
                    ->body('The database must contain both "parent_category" and "category" tables.')
                    ->send();
                return;
            }
        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Database Error')
                ->body('Error checking database tables: ' . $e->getMessage())
                ->send();
            return;
        }

        // Build command
        $command = 'categories:import';
        $params = [
            'database' => $data['database'],
            '--host' => $data['host'],
            '--port' => $data['port'],
            '--username' => $data['username'],
        ];

        if (!empty($data['password'])) {
            $params['--password'] = $data['password'];
        }

        if ($data['clear_existing']) {
            $params['--clear'] = true;
        }

        try {
            // Run the import command
            Artisan::call($command, $params);
            $output = Artisan::output();

            Notification::make()
                ->success()
                ->title('Import Completed')
                ->body('Categories have been imported successfully. Check the logs for details.')
                ->send();

            // Log the output
            \Log::info('Category import completed', ['output' => $output]);

        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Import Failed')
                ->body('Error during import: ' . $e->getMessage())
                ->send();

            \Log::error('Category import failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('import')
                ->label('Start Import')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->requiresConfirmation()
                ->modalHeading('Confirm Category Import')
                ->modalDescription('This will import categories from the old database. Are you sure you want to proceed?')
                ->modalSubmitActionLabel('Yes, Import Categories')
                ->action(fn () => $this->import()),
        ];
    }
}
