<?php

namespace App\Filament\Clusters\Settings\Pages;

use App\Filament\Clusters\Settings\SettingsCluster;
use App\Models\Category;
use App\Models\Product;
use BackedEnum;
use Filament\Actions\Action;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;

class ImportProducts extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedShoppingBag;

    protected static ?string $cluster = SettingsCluster::class;

    protected static ?string $navigationLabel = 'Import Products';

    protected static ?string $title = 'Import Products from Old Database';

    protected string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill([
            'database' => '',
            'host' => '127.0.0.1',
            'port' => '3306',
            'username' => 'root',
            'password' => '',
            'clear_existing' => false,
            'unsell_all' => true,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        $currentProductsCount = Product::count();
        $categoriesCount = Category::count();

        return $schema
            ->statePath('data')
            ->schema([
                Section::make('Database Connection')
                    ->description('Enter the connection details for the source database')
                    ->schema([
                        TextInput::make('database')
                            ->label('Database Name')
                            ->required()
                            ->placeholder('old_database_name')
                            ->helperText('The name of the database containing products table'),
                        TextInput::make('host')
                            ->label('Host')
                            ->required()
                            ->default('127.0.0.1'),
                        TextInput::make('port')
                            ->label('Port')
                            ->required()
                            ->numeric()
                            ->default('3306'),
                        TextInput::make('username')
                            ->label('Username')
                            ->required()
                            ->default('root'),
                        TextInput::make('password')
                            ->label('Password')
                            ->password()
                            ->revealable(),
                    ])
                    ->columns(2),

                Section::make('Import Options')
                    ->description("Current products: {$currentProductsCount} | Categories: {$categoriesCount}")
                    ->schema([
                        Checkbox::make('clear_existing')
                            ->label('Clear existing products before import')
                            ->helperText('⚠️ Warning: This will delete all existing products')
                            ->default(false),
                        Checkbox::make('unsell_all')
                            ->label('Mark all products as unsold')
                            ->helperText('Set sold_to to null and sold to false for all imported products')
                            ->default(true),
                    ]),
            ]);
    }

    public function import(): void
    {
        $data = $this->form->getState();

        // Check if categories exist
        $categoriesCount = Category::count();
        if ($categoriesCount === 0) {
            Notification::make()
                ->danger()
                ->title('No Categories Found')
                ->body('Please import categories first before importing products.')
                ->send();
            return;
        }

        // Test connection first
        config([
            'database.connections.test_import' => [
                'driver' => 'mysql',
                'host' => $data['host'],
                'port' => $data['port'],
                'database' => $data['database'],
                'username' => $data['username'],
                'password' => $data['password'],
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
            ]
        ]);

        try {
            DB::connection('test_import')->getPdo();
        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Connection Failed')
                ->body('Could not connect to the database: ' . $e->getMessage())
                ->send();
            return;
        }

        // Check if required table exists
        try {
            $hasProducts = DB::connection('test_import')
                ->getSchemaBuilder()
                ->hasTable('products');

            if (!$hasProducts) {
                Notification::make()
                    ->danger()
                    ->title('Missing Table')
                    ->body('The database must contain a "products" table.')
                    ->send();
                return;
            }
        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Database Error')
                ->body('Error checking database tables: ' . $e->getMessage())
                ->send();
            return;
        }

        // Build command
        $command = 'products:import';
        $params = [
            'database' => $data['database'],
            '--host' => $data['host'],
            '--port' => $data['port'],
            '--username' => $data['username'],
        ];

        if (!empty($data['password'])) {
            $params['--password'] = $data['password'];
        }

        if ($data['clear_existing']) {
            $params['--clear'] = true;
        }

        if ($data['unsell_all']) {
            $params['--unsell'] = true;
        }

        try {
            // Run the import command
            Artisan::call($command, $params);
            $output = Artisan::output();

            Notification::make()
                ->success()
                ->title('Import Completed')
                ->body('Products have been imported successfully. Check the logs for details.')
                ->send();

            // Log the output
            \Log::info('Product import completed', ['output' => $output]);

        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Import Failed')
                ->body('Error during import: ' . $e->getMessage())
                ->send();

            \Log::error('Product import failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    protected function getFormActions(): array
    {
        return [
            Action::make('import')
                ->label('Start Import')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('primary')
                ->requiresConfirmation()
                ->modalHeading('Confirm Product Import')
                ->modalDescription('This will import products from the old database. Make sure you have imported categories first. Are you sure you want to proceed?')
                ->modalSubmitActionLabel('Yes, Import Products')
                ->action(fn () => $this->import()),
        ];
    }
}
