<?php

namespace App\Filament\Clusters\Settings\Pages;

use App\Filament\Clusters\Settings\SettingsCluster;
use App\Settings\GeneralSettings;
use BackedEnum;
use Filament\Schemas\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Schema;
use Filament\Pages\Page;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Support\Icons\Heroicon;

class ManageGeneralSettings extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedCog6Tooth;

    protected static ?string $cluster = SettingsCluster::class;

    protected static ?string $navigationLabel = 'General Settings';

    protected static ?string $title = 'General Settings';

    protected string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(GeneralSettings::class);
        $this->form->fill([
            'site_name' => $settings->site_name,
            'support_username' => $settings->support_username,
            'min_deposit_usd' => $settings->min_deposit_usd,
            'max_report_days' => $settings->max_report_days,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->statePath('data')
            ->schema([
                Section::make('Site Information')
                    ->description('Basic site configuration')
                    ->schema([
                        TextInput::make('site_name')
                            ->label('Site Name')
                            ->required()
                            ->maxLength(255),
                        TextInput::make('support_username')
                            ->label('Support Telegram Username')
                            ->required()
                            ->prefix('@')
                            ->helperText('Telegram username for support (without @)'),
                    ])
                    ->columns(2),

                Section::make('System Settings')
                    ->description('Configure system behavior')
                    ->schema([
                        TextInput::make('min_deposit_usd')
                            ->label('Minimum Deposit (USD)')
                            ->required()
                            ->numeric()
                            ->prefix('$')
                            ->minValue(1)
                            ->helperText('Minimum deposit amount in USD'),
                        TextInput::make('max_report_days')
                            ->label('Report Window (Days)')
                            ->required()
                            ->numeric()
                            ->minValue(1)
                            ->maxValue(30)
                            ->helperText('Number of days users can report issues after purchase'),
                    ])
                    ->columns(2),
            ]);
    }

    public function save(): void
    {
        $data = $this->form->getState();

        $settings = app(GeneralSettings::class);
        $settings->site_name = $data['site_name'];
        $settings->support_username = $data['support_username'];
        $settings->min_deposit_usd = $data['min_deposit_usd'];
        $settings->max_report_days = $data['max_report_days'];
        $settings->save();

        Notification::make()
            ->success()
            ->title('Settings saved')
            ->body('Your settings have been saved successfully.')
            ->send();
    }

    protected function getFormActions(): array
    {
        return [
            \Filament\Actions\Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }
}
