<?php

namespace App\Filament\Clusters\Settings\Pages;

use App\Filament\Clusters\Settings\SettingsCluster;
use App\Settings\MaintenanceSettings;
use BackedEnum;
use Filament\Schemas\Components\Section;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Schema;
use Filament\Pages\Page;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Support\Icons\Heroicon;

class ManageMaintenance extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedWrenchScrewdriver;

    protected static ?string $cluster = SettingsCluster::class;

    protected static ?string $navigationLabel = 'Maintenance Mode';

    protected static ?string $title = 'Maintenance Mode';

    protected string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $settings = app(MaintenanceSettings::class);
        $this->form->fill([
            'enabled' => $settings->enabled,
            'message' => $settings->message,
            'allowed_users' => $settings->allowed_users,
        ]);
    }

    public function form(Schema $schema): Schema
    {
        return $schema
            ->statePath('data')
            ->schema([
                Section::make('Maintenance Mode')
                    ->description('Control bot availability and maintenance settings')
                    ->schema([
                        Toggle::make('enabled')
                            ->label('Enable Maintenance Mode')
                            ->helperText('When enabled, bot will show maintenance message to users')
                            ->live(),
                        Textarea::make('message')
                            ->label('Maintenance Message')
                            ->required()
                            ->rows(3)
                            ->helperText('Message shown to users during maintenance'),
                        Textarea::make('allowed_users')
                            ->label('Allowed Users (Telegram IDs)')
                            ->rows(3)
                            ->helperText('Comma-separated list of Telegram IDs that can use the bot during maintenance. Leave empty to block all users.')
                            ->placeholder('123456789, 987654321'),
                    ]),
            ]);
    }

    public function save(): void
    {
        $data = $this->form->getState();

        $settings = app(MaintenanceSettings::class);
        $settings->enabled = $data['enabled'];
        $settings->message = $data['message'];
        $settings->allowed_users = $data['allowed_users'];
        $settings->save();

        Notification::make()
            ->success()
            ->title('Maintenance settings saved')
            ->body('Your maintenance settings have been updated successfully.')
            ->send();
    }

    protected function getFormActions(): array
    {
        return [
            \Filament\Actions\Action::make('save')
                ->label('Save Settings')
                ->submit('save'),
        ];
    }
}
