<?php

namespace App\Filament\Resources\ProductReports\Pages;

use App\Filament\Resources\ProductReports\ProductReportResource;
use App\Models\ProductReport;
use App\Models\User;
use App\Services\TelegramService;
use Filament\Actions\DeleteAction;
use Filament\Actions\Action;
use Filament\Resources\Pages\EditRecord;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;

class EditProductReport extends EditRecord
{
    protected static string $resource = ProductReportResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Action::make('approve_refund')
                ->label('Approve & Refund')
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->requiresConfirmation()
                ->modalHeading('Approve Report & Process Refund')
                ->modalDescription(fn (ProductReport $record) => "This will refund \${$record->refund_amount} to the user's balance and mark the report as approved.")
                ->modalSubmitActionLabel('Approve & Refund')
                ->visible(fn (ProductReport $record) => in_array($record->status, ['pending', 'under_review']))
                ->action(function (ProductReport $record) {
                    DB::transaction(function () use ($record) {
                        // Credit user's balance
                        $record->user->increment('balance_usd', $record->refund_amount);
                        
                        // Update report
                        $record->update([
                            'status' => 'refunded',
                            'admin_id' => auth()->id(),
                            'refunded_at' => now(),
                            'admin_response' => $record->admin_response ?? 'Your report has been approved. Refund has been processed.',
                        ]);

                        // Send Telegram notification
                        $this->sendTelegramNotification($record, 'approved');
                    });

                    Notification::make()
                        ->success()
                        ->title('Report Approved')
                        ->body("Refund of \${$record->refund_amount} has been credited to user's balance.")
                        ->send();
                }),

            Action::make('deny')
                ->label('Deny Report')
                ->icon('heroicon-o-x-circle')
                ->color('danger')
                ->requiresConfirmation()
                ->modalHeading('Deny Report')
                ->modalDescription('This will deny the report and notify the user. Make sure you have provided a reason in the Admin Response field.')
                ->modalSubmitActionLabel('Deny Report')
                ->visible(fn (ProductReport $record) => in_array($record->status, ['pending', 'under_review']))
                ->action(function (ProductReport $record) {
                    $record->update([
                        'status' => 'denied',
                        'admin_id' => auth()->id(),
                        'admin_response' => $record->admin_response ?? 'Your report has been reviewed and denied.',
                    ]);

                    // Send Telegram notification
                    $this->sendTelegramNotification($record, 'denied');

                    Notification::make()
                        ->success()
                        ->title('Report Denied')
                        ->body('User has been notified of the decision.')
                        ->send();
                }),

            Action::make('request_info')
                ->label('Request More Info')
                ->icon('heroicon-o-information-circle')
                ->color('info')
                ->visible(fn (ProductReport $record) => $record->status === 'pending')
                ->action(function (ProductReport $record) {
                    $record->update([
                        'status' => 'under_review',
                        'admin_id' => auth()->id(),
                    ]);

                    Notification::make()
                        ->success()
                        ->title('Status Updated')
                        ->body('Report marked as under review.')
                        ->send();
                }),

            DeleteAction::make(),
        ];
    }

    protected function sendTelegramNotification(ProductReport $record, string $decision): void
    {
        try {
            $telegram = app(TelegramService::class);
            $user = $record->user;

            if ($decision === 'approved') {
                $text = "🔔 <b>Report Update - Approved</b>\n\n";
                $text .= "Ticket #{$record->id}\n";
                $text .= "Status: ✅ Approved\n\n";
                $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
                $text .= "<b>Admin Response:</b>\n{$record->admin_response}\n\n";
                $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
                $text .= "💰 Refund: \${$record->refund_amount}\n";
                $text .= "✅ Your balance has been credited.\n\n";
                $text .= "Check your balance: /balance";
            } else {
                $text = "🔔 <b>Report Update - Denied</b>\n\n";
                $text .= "Ticket #{$record->id}\n";
                $text .= "Status: ❌ Denied\n\n";
                $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
                $text .= "<b>Admin Response:</b>\n{$record->admin_response}\n\n";
                $text .= "If you have additional evidence, you may submit a new report.";
            }

            $keyboard = $telegram->createInlineKeyboard([
                [
                    ['text' => '📋 View Report', 'callback_data' => "viewreport:{$record->id}"],
                ],
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            $telegram->sendMessage($user->telegram_id, $text, $keyboard);
        } catch (\Exception $e) {
            \Log::error('Failed to send Telegram notification for report', [
                'report_id' => $record->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        // Set admin_id when saving
        if (empty($data['admin_id'])) {
            $data['admin_id'] = auth()->id();
        }

        return $data;
    }

    protected function afterSave(): void
    {
        // Send notification to user when report is updated
        $record = $this->record;
        
        // Check if status changed or admin response was added
        if ($record->wasChanged('status') || $record->wasChanged('admin_response')) {
            $this->sendStatusUpdateNotification($record);
        }
    }

    protected function sendStatusUpdateNotification(ProductReport $record): void
    {
        try {
            $telegram = app(TelegramService::class);
            $user = $record->user;

            $statusEmoji = match($record->status) {
                'pending' => '⏳',
                'under_review' => '🔍',
                'approved' => '✅',
                'denied' => '❌',
                'refunded' => '💰',
                default => '📋',
            };

            $text = "🔔 <b>Report Update</b>\n\n";
            $text .= "Ticket #{$record->id}\n";
            $text .= "Status: {$statusEmoji} " . ucfirst(str_replace('_', ' ', $record->status)) . "\n\n";
            
            if ($record->admin_response) {
                $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
                $text .= "<b>Admin Response:</b>\n{$record->admin_response}\n\n";
            }

            if ($record->status === 'refunded') {
                $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
                $text .= "💰 Refund: \${$record->refund_amount}\n";
                $text .= "✅ Your balance has been credited.\n\n";
            }

            $text .= "Click below to view full details:";

            $keyboard = $telegram->createInlineKeyboard([
                [
                    ['text' => '📋 View Report Details', 'callback_data' => "viewreport:{$record->id}"],
                ],
                [
                    ['text' => '📋 My Reports', 'callback_data' => 'myreports'],
                ],
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            $telegram->sendMessage($user->telegram_id, $text, $keyboard);

            Notification::make()
                ->success()
                ->title('Notification Sent')
                ->body('User has been notified via Telegram.')
                ->send();
        } catch (\Exception $e) {
            \Log::error('Failed to send status update notification', [
                'report_id' => $record->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
