<?php

namespace App\Filament\Resources\ProductReports\Schemas;

use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Textarea;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class ProductReportForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('Report Information')
                    ->schema([
                        TextInput::make('id')
                            ->label('Ticket #')
                            ->disabled()
                            ->dehydrated(false)
                            ->formatStateUsing(fn ($state) => $state ? "#{$state}" : 'New Report'),
                        Select::make('user_id')
                            ->relationship('user', 'name')
                            ->required()
                            ->disabled()
                            ->dehydrated(),
                        Select::make('order_id')
                            ->relationship('order', 'id')
                            ->required()
                            ->disabled()
                            ->dehydrated(),
                        Select::make('product_id')
                            ->relationship('product', 'title')
                            ->disabled()
                            ->dehydrated(),
                        Textarea::make('reason')
                            ->label('User\'s Issue')
                            ->required()
                            ->disabled()
                            ->dehydrated()
                            ->columnSpanFull(),
                        TextInput::make('screenshot_url')
                            ->label('Screenshot URL (Click to open)')
                            ->url()
                            ->disabled()
                            ->dehydrated()
                            ->helperText('Copy and paste this URL in your browser to view the screenshot'),
                        TextInput::make('created_at')
                            ->label('Reported At')
                            ->disabled()
                            ->dehydrated(false)
                            ->formatStateUsing(fn ($state) => $state instanceof \Carbon\Carbon ? $state->format('M d, Y H:i') : ($state ?? '—')),
                    ])
                    ->columns(2),

                Section::make('Admin Actions')
                    ->schema([
                        Select::make('status')
                            ->options([
                                'pending' => 'Pending',
                                'under_review' => 'Under Review',
                                'approved' => 'Approved',
                                'denied' => 'Denied',
                                'refunded' => 'Refunded',
                            ])
                            ->required()
                            ->default('pending'),
                        TextInput::make('refund_amount')
                            ->label('Refund Amount (USD)')
                            ->numeric()
                            ->prefix('$')
                            ->required(),
                        Textarea::make('admin_response')
                            ->label('Response to User')
                            ->placeholder('Explain your decision to the user...')
                            ->columnSpanFull()
                            ->rows(4),
                        Select::make('admin_id')
                            ->relationship('admin', 'name')
                            ->disabled()
                            ->dehydrated(),
                        DateTimePicker::make('refunded_at')
                            ->label('Refunded At')
                            ->disabled()
                            ->dehydrated(),
                    ])
                    ->columns(2),
            ]);
    }
}
