<?php

namespace App\Filament\Resources\Products\Tables;

use App\Jobs\BroadcastNewProducts;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Notifications\Notification;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Collection;

class ProductsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('seller.name')
                    ->searchable(),
                TextColumn::make('category.name')
                    ->searchable(),
                TextColumn::make('title')
                    ->searchable(),
                TextColumn::make('balance')
                    ->searchable(),
                TextColumn::make('price')
                    ->money()
                    ->sortable(),
                IconColumn::make('sold')
                    ->boolean(),
                TextColumn::make('sold_to')
                    ->searchable(),
                IconColumn::make('deleted')
                    ->boolean(),
                IconColumn::make('refund')
                    ->boolean(),
                TextColumn::make('joined')
                    ->dateTime()
                    ->sortable(),
                TextColumn::make('purchased_time')
                    ->dateTime()
                    ->sortable(),
                TextColumn::make('updated')
                    ->searchable(),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->recordActions([
                EditAction::make(),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    BulkAction::make('broadcast')
                        ->label('📢 Broadcast to Users')
                        ->icon('heroicon-o-megaphone')
                        ->color('success')
                        ->requiresConfirmation()
                        ->modalHeading('Broadcast New Products')
                        ->modalDescription('Send a notification to all users about these new products. They will be grouped by category.')
                        ->modalSubmitActionLabel('Send Broadcast')
                        ->action(function (Collection $records) {
                            $productIds = $records->pluck('id')->toArray();
                            
                            // Dispatch the broadcast job
                            BroadcastNewProducts::dispatch($productIds);
                            
                            Notification::make()
                                ->success()
                                ->title('Broadcast Queued')
                                ->body('Broadcasting ' . count($productIds) . ' products to all users.')
                                ->send();
                        }),
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
