<?php

namespace App\Filament\Widgets;

use App\Models\Category;
use App\Models\Deposit;
use App\Models\Order;
use App\Models\Product;
use App\Models\User;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;

class StatsOverviewWidget extends BaseWidget
{
    protected static ?int $sort = 1;

    protected function getStats(): array
    {
        // Total Users
        $totalUsers = User::count();
        $newUsersThisMonth = User::whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();

        // Total Products
        $totalProducts = Product::count();
        $availableProducts = Product::where('sold', false)
            ->where('deleted', false)
            ->count();
        $soldProducts = Product::where('sold', true)->count();

        // Total Categories
        $totalCategories = Category::count();
        $parentCategories = Category::whereNull('parent_id')->count();

        // Total Orders
        $totalOrders = Order::count();
        $pendingOrders = Order::where('status', 'pending')->count();
        $completedOrders = Order::where('status', 'completed')->count();

        // Revenue
        $totalRevenue = Order::where('status', 'completed')->sum('total_amount');
        $revenueThisMonth = Order::where('status', 'completed')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('total_amount');

        // Deposits
        $totalDeposits = Deposit::where('status', 'confirmed')->sum('amount_usd');
        $pendingDeposits = Deposit::where('status', 'pending')->count();

        return [
            Stat::make('Total Users', $totalUsers)
                ->description("{$newUsersThisMonth} new this month")
                ->descriptionIcon('heroicon-m-arrow-trending-up')
                ->color('success')
                ->chart([7, 12, 15, 18, 22, 25, $totalUsers]),

            Stat::make('Total Products', $totalProducts)
                ->description("{$availableProducts} available, {$soldProducts} sold")
                ->descriptionIcon('heroicon-m-shopping-bag')
                ->color('info')
                ->chart([10, 15, 20, 25, 30, 35, $totalProducts]),

            Stat::make('Categories', $totalCategories)
                ->description("{$parentCategories} parent categories")
                ->descriptionIcon('heroicon-m-folder')
                ->color('warning'),

            Stat::make('Total Orders', $totalOrders)
                ->description("{$pendingOrders} pending, {$completedOrders} completed")
                ->descriptionIcon('heroicon-m-shopping-cart')
                ->color('primary')
                ->chart([5, 10, 15, 20, 25, 30, $totalOrders]),

            Stat::make('Total Revenue', '$' . number_format($totalRevenue, 2))
                ->description('$' . number_format($revenueThisMonth, 2) . ' this month')
                ->descriptionIcon('heroicon-m-currency-dollar')
                ->color('success')
                ->chart([100, 200, 300, 400, 500, 600, $totalRevenue]),

            Stat::make('Total Deposits', '$' . number_format($totalDeposits, 2))
                ->description("{$pendingDeposits} pending confirmation")
                ->descriptionIcon('heroicon-m-banknotes')
                ->color('info'),
        ];
    }
}
