<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\BitGoService;
use App\Services\RateService;
use App\Services\ReferralService;
use App\Services\TelegramService;
use App\Models\DepositRequest;
use App\Models\Deposit;

class BitGoWebhookController extends Controller
{
    protected BitGoService $bitgo;
    protected RateService $rate;
    protected TelegramService $telegram;
    protected ReferralService $referral;

    public function __construct(BitGoService $bitgo, RateService $rate, TelegramService $telegram, ReferralService $referral)
    {
        $this->bitgo = $bitgo;
        $this->rate = $rate;
        $this->telegram = $telegram;
        $this->referral = $referral;
    }

    public function handle(Request $request)
    {
        if (!$this->bitgo->verifyWebhook($request)) {
            return response()->json(['error' => 'Invalid signature'], 401);
        }

        try {
            $txData = $this->bitgo->parseTx($request);
            
            $depositRequest = DepositRequest::where('address', $txData['address'])
                ->where('status', 'open')
                ->first();

            if (!$depositRequest) {
                return response()->json(['ok' => true]);
            }

            $deposit = Deposit::firstOrCreate(
                ['txid' => $txData['txid']],
                [
                    'user_id' => $depositRequest->user_id,
                    'deposit_request_id' => $depositRequest->id,
                    'address' => $txData['address'],
                    'amount_satoshi' => $txData['value_sats'],
                    'confirmations' => $txData['confirmations'],
                    'status' => 'pending',
                    'detected_at' => now(),
                    'metadata' => $txData['raw_data'],
                ]
            );

            if ($deposit->wasRecentlyCreated === false) {
                $deposit->update([
                    'confirmations' => $txData['confirmations'],
                ]);
            }

            $requiredConfirmations = config('services.bitgo.required_confirmations', 2);

            if ($txData['confirmations'] >= $requiredConfirmations && $deposit->credited_usd === null) {
                \DB::transaction(function () use ($deposit, $depositRequest) {
                    if ($deposit->credited_usd !== null) {
                        return;
                    }

                    $rate = $this->rate->btcUsd();
                    
                    $btc = bcdiv((string)$deposit->amount_satoshi, '100000000', 8);
                    $usd = bcmul($btc, $rate, 8);
                    $usd = floor($usd * 100) / 100;

                    $deposit->update([
                        'status' => 'confirmed',
                        'rate_usd_per_btc' => $rate,
                        'credited_usd' => $usd,
                        'credited_at' => now(),
                    ]);

                    $deposit->user->increment('balance_usd', $usd);

                    $depositRequest->update(['status' => 'used']);

                    // Process referral commission (10% of deposit)
                    $this->referral->processReferralCommission($deposit);

                    $text = "✅ <b>Deposit Confirmed!</b>\n\n";
                    $text .= "Amount: {$btc} BTC\n";
                    $text .= "Credited: \${$usd} USD\n";
                    $text .= "Rate: \${$rate}/BTC\n\n";
                    $text .= "New balance: \${$deposit->user->fresh()->balance_usd}";

                    $keyboard = $this->telegram->createInlineKeyboard([
                        [
                            ['text' => '🛍 Browse Products', 'callback_data' => 'browse'],
                        ],
                    ]);

                    $this->telegram->sendMessage(
                        (int)$deposit->user->telegram_id,
                        $text,
                        $keyboard
                    );
                });
            }

            return response()->json(['ok' => true]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Processing failed'], 500);
        }
    }
}
