<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\TelegramService;
use App\Services\QrService;
use App\Models\User;
use App\Models\Category;
use App\Models\Product;
use App\Models\DepositRequest;
use App\Models\Deposit;
use App\Models\Order;
use App\Services\BitGoService;

class BotWebhookController extends Controller
{
    protected TelegramService $telegram;
    protected QrService $qr;
    protected BitGoService $bitgo;

    public function __construct(TelegramService $telegram, QrService $qr, BitGoService $bitgo)
    {
        $this->telegram = $telegram;
        $this->qr = $qr;
        $this->bitgo = $bitgo;
    }

    public function handle(\App\Http\Requests\TelegramWebhookRequest $request)
    {
        $update = $request->validated();

        if (isset($update['message'])) {
            $this->handleMessage($update['message']);
        } elseif (isset($update['callback_query'])) {
            $this->handleCallbackQuery($update['callback_query']);
        }

        return response()->json(['ok' => true]);
    }

    protected function handleMessage(array $message)
    {
        $chatId = $message['chat']['id'];
        $text = $message['text'] ?? '';
        $telegramId = (string)$message['from']['id'];
        $username = $message['from']['username'] ?? null;

        $user = User::firstOrCreate(
            ['telegram_id' => $telegramId],
            [
                'name' => $message['from']['first_name'] ?? 'User',
                'username' => $username,
                'role' => 'user',
            ]
        );

        if ($text === '/start') {
            $this->handleStart($chatId, $user);
        } elseif ($text === '/balance') {
            $this->handleBalance($chatId, $user);
        } elseif ($text === '/rules') {
            $this->handleRules($chatId);
        } elseif ($text === '/help') {
            $this->handleHelp($chatId);
        } elseif ($text === '/history') {
            $this->handleHistory($chatId, $user);
        }
    }

    protected function handleCallbackQuery(array $callbackQuery)
    {
        $chatId = $callbackQuery['message']['chat']['id'];
        $messageId = $callbackQuery['message']['message_id'];
        $data = $callbackQuery['data'];
        $telegramId = (string)$callbackQuery['from']['id'];

        $user = User::where('telegram_id', $telegramId)->first();

        if (!$user) {
            return;
        }

        $this->telegram->answerCallbackQuery($callbackQuery['id']);

        $parts = explode(':', $data);
        $action = $parts[0];

        match ($action) {
            'bal' => $this->handleBalance($chatId, $user, $messageId),
            'dep' => $this->handleDeposit($chatId, $user, $messageId),
            'depstatus' => $this->handleDepositStatus($chatId, $user, $messageId, (int)($parts[1] ?? 0)),
            'browse' => $this->handleBrowse($chatId, $messageId),
            'cat' => $this->handleCategory($chatId, $messageId, (int)$parts[1], (int)($parts[2] ?? 0)),
            'prd' => $this->handleProduct($chatId, $messageId, (int)$parts[1]),
            'buyconfirm' => $this->handleBuyConfirm($chatId, $user, $messageId, (int)$parts[1]),
            'buy' => $this->handleBuy($chatId, $user, $messageId, (int)$parts[1]),
            'pur' => $this->handlePurchases($chatId, $user, $messageId, (int)($parts[1] ?? 0)),
            'history' => $this->handleHistory($chatId, $user, $messageId),
            'help' => $this->handleHelp($chatId, $messageId),
            'menu' => $this->handleStart($chatId, $user, $messageId),
            default => null,
        };
    }

    protected function handleStart(int $chatId, User $user, ?int $messageId = null)
    {
        $isNewUser = $user->wasRecentlyCreated;
        
        if ($isNewUser) {
            $text = "👋 <b>Welcome to ToolStore!</b>\n\n";
            $text .= "Your trusted marketplace for digital products.\n\n";
            $text .= "🎯 <b>What we offer:</b>\n";
            $text .= "• Digital downloads & tools\n";
            $text .= "• API credits & services\n";
            $text .= "• Instant delivery\n";
            $text .= "• Secure Bitcoin payments\n\n";
            $text .= "💡 <b>How it works:</b>\n";
            $text .= "1️⃣ Deposit Bitcoin → Get USD balance\n";
            $text .= "2️⃣ Browse our products\n";
            $text .= "3️⃣ Purchase & receive instantly\n\n";
            $text .= "Ready to start shopping?";
        } else {
            $totalProducts = Product::where('sold', false)->where('deleted', false)->count();
            $text = "🏪 <b>Welcome back to ToolStore!</b>\n\n";
            $text .= "Your balance: <b>\${$user->balance_usd}</b>\n";
            $text .= "Available products: <b>{$totalProducts}</b>\n\n";
            $text .= "What would you like to do?";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '🛍 Browse Products', 'callback_data' => 'browse'],
                ['text' => '💰 Deposit BTC', 'callback_data' => 'dep:new'],
            ],
            [
                ['text' => '💵 My Balance', 'callback_data' => 'bal'],
                ['text' => '📦 My Purchases', 'callback_data' => 'pur:0'],
            ],
            [
                ['text' => '📜 Rules & Terms', 'callback_data' => 'rules'],
                ['text' => '❓ Help', 'callback_data' => 'help'],
            ],
        ]);

        if ($messageId) {
            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->telegram->sendMessage($chatId, $text, $keyboard);
        }
    }

    protected function handleBalance(int $chatId, User $user, ?int $messageId = null)
    {
        $totalDeposits = Deposit::where('user_id', $user->id)
            ->where('status', 'confirmed')
            ->sum('credited_usd');
        
        $totalSpent = Order::where('user_id', $user->id)
            ->where('status', 'paid')
            ->sum('total_usd');
        
        $pendingDeposits = DepositRequest::where('user_id', $user->id)
            ->where('status', 'open')
            ->count();

        $text = "💵 <b>Your Balance</b>\n\n";
        $text .= "Current Balance: <b>\${$user->balance_usd}</b>\n\n";
        $text .= "📊 <b>Statistics:</b>\n";
        $text .= "Total Deposited: \${$totalDeposits}\n";
        $text .= "Total Spent: \${$totalSpent}\n";
        
        if ($pendingDeposits > 0) {
            $text .= "\n⏳ Pending Deposits: {$pendingDeposits}\n";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '💰 Deposit BTC', 'callback_data' => 'dep:new'],
                ['text' => '📜 History', 'callback_data' => 'history'],
            ],
            [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        if ($messageId) {
            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->telegram->sendMessage($chatId, $text, $keyboard);
        }
    }

    protected function handleDeposit(int $chatId, User $user, ?int $messageId = null)
    {
        $label = "user_{$user->id}_deposit_" . time();
        
        try {
            $addressData = $this->bitgo->createAddress($label);
            
            $depositRequest = DepositRequest::create([
                'user_id' => $user->id,
                'provider' => 'bitgo',
                'chain' => 'btc',
                'address' => $addressData['address'],
                'label' => $label,
                'status' => 'open',
            ]);

            $qrDataUri = $this->qr->generateBitcoinQr($addressData['address']);

            $text = "💰 <b>Deposit Bitcoin</b>\n\n";
            $text .= "Send BTC to this address:\n";
            $text .= "<code>{$addressData['address']}</code>\n\n";
            $text .= "You'll be credited in USD after 2 confirmations at the market rate.\n\n";
            $text .= "⚠️ Only send Bitcoin to this address!";

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->sendPhoto($chatId, $qrDataUri, $text, $keyboard);
        } catch (\Exception $e) {
            $text = "❌ Failed to create deposit address. Please try again later.";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            if ($messageId) {
                $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
            } else {
                $this->telegram->sendMessage($chatId, $text, $keyboard);
            }
        }
    }

    protected function handleBrowse(int $chatId, ?int $messageId = null)
    {
        $categories = Category::whereNull('parent_id')
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        $text = "🛍 <b>Browse Categories</b>\n\nSelect a category:";

        $buttons = [];
        foreach ($categories as $category) {
            $buttons[] = [
                ['text' => $category->name, 'callback_data' => "cat:{$category->id}:0"],
            ];
        }

        $buttons[] = [
            ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
        ];

        $keyboard = $this->telegram->createInlineKeyboard($buttons);

        if ($messageId) {
            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->telegram->sendMessage($chatId, $text, $keyboard);
        }
    }

    protected function handleCategory(int $chatId, int $messageId, int $categoryId, int $page = 0)
    {
        $category = Category::find($categoryId);

        if (!$category) {
            return;
        }

        $subcategories = Category::where('parent_id', $categoryId)
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->get();

        if ($subcategories->isNotEmpty()) {
            $text = "📁 <b>{$category->name}</b>\n\nSelect a subcategory:";

            $buttons = [];
            foreach ($subcategories as $subcategory) {
                $buttons[] = [
                    ['text' => $subcategory->name, 'callback_data' => "cat:{$subcategory->id}:0"],
                ];
            }

            $buttons[] = [
                ['text' => '⬅️ Back', 'callback_data' => 'browse'],
            ];

            $keyboard = $this->telegram->createInlineKeyboard($buttons);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $perPage = 10;
            $products = Product::where('category_id', $categoryId)
                ->where('sold', false)
                ->where('deleted', false)
                ->skip($page * $perPage)
                ->take($perPage)
                ->get();

            $total = Product::where('category_id', $categoryId)
                ->where('sold', false)
                ->where('deleted', false)
                ->count();

            $text = "📦 <b>{$category->name}</b>\n\n";
            
            if ($products->isEmpty()) {
                $text .= "No products available.";
            } else {
                $text .= "Available products:";
            }

            $buttons = [];
            foreach ($products as $product) {
                $buttons[] = [
                    ['text' => "{$product->title} - \${$product->price}", 'callback_data' => "prd:{$product->id}"],
                ];
            }

            if ($total > ($page + 1) * $perPage) {
                $buttons[] = [
                    ['text' => '➡️ Next', 'callback_data' => "cat:{$categoryId}:" . ($page + 1)],
                ];
            }

            $buttons[] = [
                ['text' => '⬅️ Back', 'callback_data' => 'browse'],
            ];

            $keyboard = $this->telegram->createInlineKeyboard($buttons);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        }
    }

    protected function handleProduct(int $chatId, int $messageId, int $productId)
    {
        $product = Product::find($productId);

        if (!$product || $product->sold || $product->deleted) {
            $text = "❌ Product not available.";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '⬅️ Back', 'callback_data' => 'browse'],
                ],
            ]);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
            return;
        }

        $text = "📦 <b>{$product->title}</b>\n\n";
        $text .= "💵 Price: <b>\${$product->price}</b>\n\n";
        
        if ($product->short_description) {
            $text .= "{$product->short_description}\n\n";
        }

        if ($product->balance) {
            $text .= "💰 Includes: \${$product->balance} balance\n\n";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '🛒 Buy Now', 'callback_data' => "buyconfirm:{$product->id}"],
            ],
            [
                ['text' => '⬅️ Back', 'callback_data' => "cat:{$product->category_id}:0"],
            ],
        ]);

        $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
    }

    protected function handleBuy(int $chatId, User $user, int $messageId, int $productId)
    {
        $product = Product::find($productId);

        if (!$product || $product->sold || $product->deleted) {
            $text = "❌ Product not available.";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
            return;
        }

        if ($user->balance_usd < $product->price) {
            $needed = $product->price - $user->balance_usd;
            $text = "❌ <b>Insufficient Balance</b>\n\n";
            $text .= "You need \${$needed} more.\n\n";
            $text .= "Current balance: \${$user->balance_usd}\n";
            $text .= "Product price: \${$product->price}";

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '💰 Deposit', 'callback_data' => 'dep:new'],
                ],
                [
                    ['text' => '⬅️ Back', 'callback_data' => "prd:{$product->id}"],
                ],
            ]);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
            return;
        }

        \DB::transaction(function () use ($user, $product, $chatId, $messageId) {
            $order = Order::create([
                'user_id' => $user->id,
                'total_usd' => $product->price,
                'status' => 'paid',
            ]);

            $order->items()->create([
                'product_id' => $product->id,
                'qty' => 1,
                'unit_price_usd' => $product->price,
                'line_total_usd' => $product->price,
                'snapshot' => $product->toArray(),
            ]);

            $user->decrement('balance_usd', $product->price);

            $product->update([
                'sold' => true,
                'sold_to' => $user->telegram_id,
                'purchased_time' => now(),
            ]);

            $text = "✅ <b>Purchase Successful!</b>\n\n";
            $text .= "Product: {$product->title}\n";
            $text .= "Amount: \${$product->price}\n\n";
            
            if ($product->details) {
                $text .= "📄 <b>Product Details:</b>\n\n";
                $text .= $product->details . "\n\n";
            }

            $text .= "New balance: \${$user->fresh()->balance_usd}";

            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '📦 My Purchases', 'callback_data' => 'pur:0'],
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        });
    }

    protected function handlePurchases(int $chatId, User $user, int $messageId, int $page = 0)
    {
        $perPage = 5;
        $orders = Order::where('user_id', $user->id)
            ->where('status', 'paid')
            ->with('items.product')
            ->orderBy('created_at', 'desc')
            ->skip($page * $perPage)
            ->take($perPage)
            ->get();

        $total = Order::where('user_id', $user->id)
            ->where('status', 'paid')
            ->count();

        $text = "📦 <b>My Purchases</b>\n\n";

        if ($orders->isEmpty()) {
            $text .= "You haven't made any purchases yet.";
        } else {
            foreach ($orders as $order) {
                foreach ($order->items as $item) {
                    $product = $item->product;
                    $text .= "• {$product->title} - \${$item->unit_price_usd}\n";
                    $text .= "  Date: {$order->created_at->format('Y-m-d H:i')}\n\n";
                }
            }
        }

        $buttons = [];

        if ($total > ($page + 1) * $perPage) {
            $buttons[] = [
                ['text' => '➡️ Next', 'callback_data' => "pur:" . ($page + 1)],
            ];
        }

        $buttons[] = [
            ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
        ];

        $keyboard = $this->telegram->createInlineKeyboard($buttons);

        $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
    }

    protected function handleRules(int $chatId)
    {
        $text = "📜 <b>Rules & Terms</b>\n\n";
        $text .= "• Lawful goods only\n";
        $text .= "• No reselling credentials or other people's data\n";
        $text .= "• All items are moderated\n";
        $text .= "• Refunds for non-delivery only\n";
        $text .= "• Misuse will result in ban\n\n";
        $text .= "By using this service, you agree to these terms.";

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        $this->telegram->sendMessage($chatId, $text, $keyboard);
    }

    protected function handleHelp(int $chatId, ?int $messageId = null)
    {
        $text = "❓ <b>Help & Commands</b>\n\n";
        $text .= "<b>Available Commands:</b>\n";
        $text .= "/start - Main menu\n";
        $text .= "/balance - Check your balance\n";
        $text .= "/history - Transaction history\n";
        $text .= "/help - Show this help\n";
        $text .= "/rules - Terms and conditions\n\n";
        
        $text .= "<b>How to use:</b>\n";
        $text .= "1️⃣ <b>Deposit:</b> Click 'Deposit BTC' to get a unique Bitcoin address\n";
        $text .= "2️⃣ <b>Browse:</b> Explore categories and products\n";
        $text .= "3️⃣ <b>Purchase:</b> Buy products with your USD balance\n";
        $text .= "4️⃣ <b>Receive:</b> Get instant access to your purchase\n\n";
        
        $text .= "<b>Need support?</b>\n";
        $text .= "Contact: @YourSupportUsername";

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '🛍 Browse Products', 'callback_data' => 'browse'],
                ['text' => '💰 Deposit', 'callback_data' => 'dep:new'],
            ],
            [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        if ($messageId) {
            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->telegram->sendMessage($chatId, $text, $keyboard);
        }
    }

    protected function handleHistory(int $chatId, User $user, ?int $messageId = null)
    {
        $deposits = Deposit::where('user_id', $user->id)
            ->where('status', 'confirmed')
            ->orderBy('credited_at', 'desc')
            ->take(5)
            ->get();
        
        $orders = Order::where('user_id', $user->id)
            ->where('status', 'paid')
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        $text = "📜 <b>Transaction History</b>\n\n";
        
        if ($deposits->isNotEmpty()) {
            $text .= "💰 <b>Recent Deposits:</b>\n";
            foreach ($deposits as $deposit) {
                $btc = \App\Services\BitGoService::satoshisToBtc($deposit->amount_satoshi);
                $date = $deposit->credited_at->format('M d, H:i');
                $text .= "• {$btc} BTC → \${$deposit->credited_usd} ({$date})\n";
            }
            $text .= "\n";
        }
        
        if ($orders->isNotEmpty()) {
            $text .= "🛍 <b>Recent Purchases:</b>\n";
            foreach ($orders as $order) {
                $date = $order->created_at->format('M d, H:i');
                $text .= "• \${$order->total_usd} ({$date})\n";
            }
            $text .= "\n";
        }
        
        if ($deposits->isEmpty() && $orders->isEmpty()) {
            $text .= "No transactions yet.\n\n";
            $text .= "Start by making a deposit!";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '💵 Balance', 'callback_data' => 'bal'],
                ['text' => '💰 Deposit', 'callback_data' => 'dep:new'],
            ],
            [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        if ($messageId) {
            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
        } else {
            $this->telegram->sendMessage($chatId, $text, $keyboard);
        }
    }

    protected function handleDepositStatus(int $chatId, User $user, int $messageId, int $requestId)
    {
        $depositRequest = DepositRequest::find($requestId);
        
        if (!$depositRequest || $depositRequest->user_id !== $user->id) {
            $text = "❌ Deposit request not found.";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);
            
            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
            return;
        }

        $deposits = Deposit::where('deposit_request_id', $requestId)->get();
        
        $text = "📊 <b>Deposit Status</b>\n\n";
        $text .= "Address: <code>{$depositRequest->address}</code>\n";
        $text .= "Status: <b>" . ucfirst($depositRequest->status) . "</b>\n\n";
        
        if ($deposits->isNotEmpty()) {
            $text .= "<b>Transactions:</b>\n";
            foreach ($deposits as $deposit) {
                $btc = \App\Services\BitGoService::satoshisToBtc($deposit->amount_satoshi);
                $text .= "• {$btc} BTC\n";
                $text .= "  Confirmations: {$deposit->confirmations}/2\n";
                $text .= "  Status: " . ucfirst($deposit->status) . "\n";
                
                if ($deposit->credited_usd) {
                    $text .= "  Credited: \${$deposit->credited_usd}\n";
                }
                $text .= "\n";
            }
        } else {
            $text .= "⏳ Waiting for transaction...\n";
            $text .= "Send Bitcoin to the address above.";
        }

        $keyboard = $this->telegram->createInlineKeyboard([
            [
                ['text' => '🔄 Refresh', 'callback_data' => "depstatus:{$requestId}"],
            ],
            [
                ['text' => '💵 Balance', 'callback_data' => 'bal'],
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ],
        ]);

        $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
    }

    protected function handleBuyConfirm(int $chatId, User $user, int $messageId, int $productId)
    {
        $product = Product::find($productId);

        if (!$product || $product->sold || $product->deleted) {
            $text = "❌ Product not available.";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
                ],
            ]);

            $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
            return;
        }

        $text = "🛒 <b>Confirm Purchase</b>\n\n";
        $text .= "Product: <b>{$product->title}</b>\n";
        $text .= "Price: <b>\${$product->price}</b>\n\n";
        $text .= "Your current balance: \${$user->balance_usd}\n";
        
        if ($user->balance_usd >= $product->price) {
            $newBalance = $user->balance_usd - $product->price;
            $text .= "Balance after purchase: \${$newBalance}\n\n";
            $text .= "✅ Proceed with purchase?";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '✅ Confirm Purchase', 'callback_data' => "buy:{$product->id}"],
                ],
                [
                    ['text' => '❌ Cancel', 'callback_data' => "prd:{$product->id}"],
                ],
            ]);
        } else {
            $needed = $product->price - $user->balance_usd;
            $text .= "\n❌ Insufficient balance!\n";
            $text .= "You need \${$needed} more.";
            
            $keyboard = $this->telegram->createInlineKeyboard([
                [
                    ['text' => '💰 Deposit', 'callback_data' => 'dep:new'],
                ],
                [
                    ['text' => '⬅️ Back', 'callback_data' => "prd:{$product->id}"],
                ],
            ]);
        }

        $this->telegram->editMessageText($chatId, $messageId, $text, $keyboard);
    }
}
