<?php

namespace App\Jobs;

use App\Models\Product;
use App\Models\User;
use App\Services\TelegramService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Collection;

class BroadcastNewProducts implements ShouldQueue
{
    use Queueable;

    /**
     * Create a new job instance.
     */
    public function __construct(
        public array $productIds
    ) {}

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $telegram = app(TelegramService::class);

        // Get products with their categories
        $products = Product::with('category')
            ->whereIn('id', $this->productIds)
            ->get();

        if ($products->isEmpty()) {
            return;
        }

        // Group products by category
        $groupedProducts = $products->groupBy('category.name');

        // Build broadcast message
        $message = "";


        // Get all users who have accepted terms
        $users = User::where('terms_accepted', true)
            ->whereNotNull('telegram_id')
            ->get();

        foreach ($groupedProducts as $categoryName => $categoryProducts) {
            $count = $categoryProducts->count();
            // Get first user's language as default (or 'ru')
            $lang = $users->first()?->language ?? 'ru';

            if (empty($message)) {
                $message = __('bot.new_products_added', [], $lang) . "\n\n";
            }

            $items = $count === 1 ? __('bot.item_singular', [], $lang) : __('bot.items_plural', [], $lang);
            $message .= __('bot.added_items_category', [
                'count' => $count,
                'items' => $items,
                'category' => $categoryName
            ], $lang) . "\n";
        }

        $lang = $users->first()?->language ?? 'ru';
        $message .= "\n" . __('bot.browse_updated_catalog', [], $lang);



        // Send broadcast to all users
        foreach ($users as $user) {
            try {
                $telegram->sendMessage((int)$user->telegram_id, $message);

                // Small delay to avoid rate limiting
                usleep(50000); // 50ms delay
            } catch (\Exception $e) {
                \Log::error('Broadcast failed for user', [
                    'user_id' => $user->id,
                    'telegram_id' => $user->telegram_id,
                    'error' => $e->getMessage()
                ]);
            }
        }

    }
}
