<?php

declare(strict_types=1);

namespace App\Jobs;

use App\Actions\Telegram\ProcessStartCommand;
use App\Actions\Telegram\ProcessBalanceCommand;
use App\Actions\Telegram\ProcessHelpCommand;
use App\Actions\Telegram\ProcessHistoryCommand;
use App\Actions\Telegram\ShowBalanceAction;
use App\Actions\Telegram\CreateDepositAction;
use App\Actions\Telegram\BrowseProductsAction;
use App\Actions\Telegram\ShowCategoryProductsAction;
use App\Actions\Telegram\ShowProductDetailsAction;
use App\Actions\Telegram\PurchaseProductAction;
use App\Actions\Telegram\ShowMainMenuAction;
use App\Actions\Telegram\ShowLanguageSelectionAction;
use App\Actions\Telegram\ShowReferralInfoAction;
use App\Actions\Telegram\ReportHandlers;
use App\Bot\Data\TelegramUpdateData;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Cache;

class HandleTelegramUpdate implements ShouldQueue
{
    use Queueable;

    public function __construct(
        public TelegramUpdateData $update
    ) {}

    public function handle(): void
    {
        if ($this->update->message) {
            $this->handleMessage($this->update->message);
        }

        if ($this->update->callback_query) {
            $this->handleCallback($this->update->callback_query);
        }
    }

    private function handleMessage($message): void
    {
        // Check maintenance mode
        $maintenance = app(\App\Settings\MaintenanceSettings::class);
        if ($maintenance->enabled) {
            $allowedUsers = array_filter(array_map('trim', explode(',', $maintenance->allowed_users)));
            $telegramId = (string)$message->from->id;

            if (!in_array($telegramId, $allowedUsers)) {
                $telegram = app(\App\Services\TelegramService::class);
                $telegram->sendMessage($message->chat->id, $maintenance->message);
                return;
            }
        }

        // Check if user is in a conversation state (e.g., reporting an issue)
        $user = \App\Models\User::where('telegram_id', (string)$message->from->id)->first();

        if ($user) {
            $state = Cache::get("report_state:{$user->id}");

            if ($state) {
                $reportHandler = app(ReportHandlers::class);

                if ($state['step'] === 'awaiting_reason') {
                    $reportHandler->handleReportReason($message, $user, $message->text);
                    return;
                } elseif ($state['step'] === 'awaiting_screenshot') {
                    $reportHandler->handleReportScreenshot($message, $user, $message->text);
                    return;
                }
            }
        }

        // Handle /start command (no terms check needed)
        if ($message->text === '/start') {
            app(ProcessStartCommand::class)->execute($message);
            return;
        }

        // Check if user has accepted terms for all other commands
        if ($user && !$user->terms_accepted) {
            $telegram = app(\App\Services\TelegramService::class);
            $lang = $user->language ?? 'ru';
            $text = __('bot.terms_required', [], $lang);
            $telegram->sendMessage($message->chat->id, $text);
            return;
        }

        // Handle other commands
        match ($message->text) {
            '/balance' => app(ProcessBalanceCommand::class)->execute($message),
            '/deposit' => $this->handleDepositCommand($message),
            '/help' => app(ProcessHelpCommand::class)->execute($message),
            '/history' => app(ProcessHistoryCommand::class)->execute($message),
            '/rules' => $this->handleRulesCommand($message),
            '/myreports' => $this->handleMyReportsCommand($message),
            default => null,
        };
    }

    private function handleDepositCommand($message): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$message->from->id)->first();
        $lang = $user?->language ?? 'ru';

        $text = __('bot.deposit_bitcoin_title', [], $lang) . "\n\n";
        $text .= __('bot.deposit_click_button', [], $lang) . "\n\n";
        $text .= __('bot.deposit_receive_qr', [], $lang);

        $keyboard = $telegram->createInlineKeyboard([
            [
                ['text' => __('bot.deposit_btc', [], $lang), 'callback_data' => 'dep:new'],
            ],
            [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ],
        ]);

        $telegram->sendMessage($message->chat->id, $text, $keyboard);
    }

    private function handleRulesCommand($message): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$message->from->id)->first();
        $lang = $user?->language ?? 'ru';

        $text = __('bot.rules_text', [], $lang);

        $keyboard = $telegram->createInlineKeyboard([
            [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ],
        ]);

        $telegram->sendMessage($message->chat->id, $text, $keyboard);
    }

    private function handleMyReportsCommand($message): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$message->from->id)->first();
        $lang = $user?->language ?? 'ru';
        if (!$user) {
            return;
        }

        $reports = \App\Models\ProductReport::where('user_id', $user->id)
            ->with(['order', 'product'])
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        if ($reports->isEmpty()) {
            $text = __('bot.my_reports_title', [], $lang) . "\n\n";
            $text .= __('bot.no_reports_yet', [], $lang) . "\n\n";
            $text .= __('bot.report_from_orders', [], $lang);

            $keyboard = $telegram->createInlineKeyboard([
                [
                    ['text' => __('bot.my_purchases', [], $lang), 'callback_data' => 'pur:0'],
                ],
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ]);
        } else {
            $text = __('bot.my_reports_title', [], $lang) . "\n\n";
            $text .= __('bot.click_report_details', [], $lang) . "\n\n";

            $buttons = [];
            foreach ($reports as $report) {
                $statusIcon = match($report->status) {
                    'pending' => '⏳',
                    'under_review' => '🔍',
                    'approved' => '✅',
                    'denied' => '❌',
                    'refunded' => '💰',
                    default => '📋',
                };

                $productName = $report->product ? $report->product->title : 'Product';
                $buttons[] = [
                    ['text' => "{$statusIcon} #{$report->id} - {$productName} - \${$report->refund_amount}", 'callback_data' => "viewreport:{$report->id}"],
                ];
            }

            $buttons[] = [
                ['text' => '🏠 Main Menu', 'callback_data' => 'menu'],
            ];

            $keyboard = $telegram->createInlineKeyboard($buttons);
        }

        $telegram->sendMessage($message->chat->id, $text, $keyboard);
    }

    private function handleCallback($callback): void
    {
        $parts = explode(':', $callback->data);
        $action = $parts[0];

        // Handle terms acceptance (no terms check needed)
        if ($action === 'terms') {
            $this->handleTermsAction($callback, $parts[1] ?? '');
            return;
        }

        // Check if user has accepted terms for all other actions
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();
        if ($user && !$user->terms_accepted) {
            $telegram = app(\App\Services\TelegramService::class);
            $lang = $user->language ?? 'ru';
            $telegram->answerCallbackQuery($callback->id);

            $text = __('bot.terms_required', [], $lang);

            $telegram->sendMessage($callback->message->chat->id, $text);
            return;
        }

        match ($action) {
            'bal' => app(ShowBalanceAction::class)->execute($callback),
            'dep' => app(CreateDepositAction::class)->execute($callback),
            'browse' => app(BrowseProductsAction::class)->execute($callback),
            'cat' => app(ShowCategoryProductsAction::class)->execute($callback, (int)($parts[1] ?? 0), (int)($parts[2] ?? 0)),
            'prd' => app(ShowProductDetailsAction::class)->execute($callback, (int)($parts[1] ?? 0)),
            'buy' => app(PurchaseProductAction::class)->execute($callback, (int)($parts[1] ?? 0)),
            'buyconfirm' => app(PurchaseProductAction::class)->execute($callback, (int)($parts[1] ?? 0)),
            'menu' => app(ShowMainMenuAction::class)->execute($callback),
            'language' => app(ShowLanguageSelectionAction::class)->execute($callback),
            'lang' => app(ShowLanguageSelectionAction::class)->setLanguage($callback, $parts[1] ?? 'en'),
            'referral' => app(ShowReferralInfoAction::class)->execute($callback),
            'help' => app(ProcessHelpCommand::class)->execute($this->convertCallbackToMessage($callback)),
            'rules' => $this->handleRules($callback),
            'pur' => $this->handlePurchases($callback, (int)($parts[1] ?? 0)),
            'vieworder' => $this->handleViewOrderDetails($callback, (int)($parts[1] ?? 0)),
            'report' => app(ReportHandlers::class)->handleStartReport($callback, (int)($parts[1] ?? 0)),
            'reportskip' => $this->handleReportSkip($callback),
            'myreports' => app(ReportHandlers::class)->handleMyReports($callback),
            'viewreport' => app(ReportHandlers::class)->handleViewReport($callback, (int)($parts[1] ?? 0)),
            'history' => app(ProcessHistoryCommand::class)->execute($this->convertCallbackToMessage($callback)),
            'depstatus' => $this->handleDepositStatus($callback, (int)($parts[1] ?? 0)),
            default => null,
        };
    }

    private function handleTermsAction($callback, string $action): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();

        if (!$user) {
            return;
        }

        $lang = $user->language ?? 'ru';

        if ($action === 'accept') {
            // User accepted terms
            $user->update([
                'terms_accepted' => true,
                'terms_accepted_at' => now(),
            ]);

            $text = __('bot.terms_accepted_title', [], $lang) . "\n\n";
            $text .= __('bot.terms_accepted_thank_you', [], $lang) . "\n\n";
            $text .= __('bot.terms_accepted_can_use', [], $lang) . "\n\n";
            $text .= __('bot.terms_accepted_start', [], $lang);

            $telegram->answerCallbackQuery($callback->id);
            $telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                $text
            );
        } else {
            // User declined terms
            $text = __('bot.terms_declined_title', [], $lang) . "\n\n";
            $text .= __('bot.terms_declined_must_accept', [], $lang) . "\n\n";
            $text .= __('bot.terms_declined_try_again', [], $lang);

            $telegram->answerCallbackQuery($callback->id);
            $telegram->editMessageText(
                $callback->message->chat->id,
                $callback->message->message_id,
                $text
            );
        }
    }

    private function convertCallbackToMessage($callback)
    {
        return new \App\Bot\Data\TelegramMessageData(
            message_id: $callback->message->message_id,
            chat: $callback->message->chat,
            from: $callback->from,
            text: null
        );
    }

    private function handleRules($callback): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();
        $lang = $user?->language ?? 'ru';

        $text = __('bot.rules_text', [], $lang);

        $keyboard = $telegram->createInlineKeyboard([
            [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ],
        ]);

        $telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }

    private function handlePurchases($callback, int $page = 0): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();
        $lang = $user?->language ?? 'ru';
        if (!$user) {
            return;
        }

        $perPage = 5;
        $orders = \App\Models\Order::where('user_id', $user->id)
            ->where('status', 'paid')
            ->with('items.product')
            ->orderBy('created_at', 'desc')
            ->skip($page * $perPage)
            ->take($perPage)
            ->get();

        $totalOrders = \App\Models\Order::where('user_id', $user->id)
            ->where('status', 'paid')
            ->count();

        if ($orders->isEmpty()) {
            $text = __('bot.my_purchases_title', [], $lang) . "\n\n";
            $text .= __('bot.no_purchases_yet', [], $lang) . "\n\n";
            $text .= __('bot.browse_to_start', [], $lang);

            $buttons = [
                [
                    ['text' => __('bot.browse_products', [], $lang), 'callback_data' => 'browse'],
                ],
                [
                    ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                ],
            ];
        } else {
            $text = __('bot.my_purchases_title', [], $lang) . "\n\n";
            $text .= __('bot.click_order_details', [], $lang) . "\n\n";

            $buttons = [];
            foreach ($orders as $order) {
                $date = $order->created_at->format('M d, Y');
                $itemCount = $order->items->count();
                $itemText = $itemCount === 1 ? __('bot.item_count', [], $lang) : __('bot.items_count', ['count' => $itemCount], $lang);

                $buttons[] = [
                    ['text' => "📋 Order #{$order->id} - {$date} - \${$order->total_usd} ({$itemText})", 'callback_data' => "vieworder:{$order->id}"],
                ];
            }

            // Pagination
            $navButtons = [];
            if ($page > 0) {
                $navButtons[] = ['text' => __('bot.previous', [], $lang), 'callback_data' => "pur:" . ($page - 1)];
            }
            if (($page + 1) * $perPage < $totalOrders) {
                $navButtons[] = ['text' => __('bot.next', [], $lang), 'callback_data' => "pur:" . ($page + 1)];
            }
            if (!empty($navButtons)) {
                $buttons[] = $navButtons;
            }

            $buttons[] = [
                ['text' => __('bot.browse_products', [], $lang), 'callback_data' => 'browse'],
            ];
            $buttons[] = [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ];
        }

        $keyboard = $telegram->createInlineKeyboard($buttons);

        $telegram->editMessageText(
            $callback->message->chat->id,
            $callback->message->message_id,
            $text,
            $keyboard
        );
    }

    private function handleViewOrderDetails($callback, int $orderId): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();
        $lang = $user?->language ?? 'ru';

        if (!$user) {
            return;
        }

        $order = \App\Models\Order::where('id', $orderId)
            ->where('user_id', $user->id)
            ->with('items.product')
            ->first();

        if (!$order) {
            $text = __('bot.order_not_found', [], $lang);
            $keyboard = $telegram->createInlineKeyboard([[
                ['text' => __('bot.back_to_purchases', [], $lang), 'callback_data' => 'pur:0'],
            ]]);
        } else {
            $text = __('bot.order_title', ['id' => $order->id], $lang) . "\n\n";
            $text .= __('bot.order_date', ['date' => $order->created_at->format('M d, Y H:i')], $lang) . "\n";
            $text .= __('bot.order_total', ['total' => $order->total_usd], $lang) . "\n";
            $text .= __('bot.order_status', ['status' => ucfirst($order->status)], $lang) . "\n\n";
            $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";

            foreach ($order->items as $item) {
                $product = $item->product;
                $snapshot = $item->snapshot;

                $text .= "🔹 <b>" . ($product ? $product->title : $snapshot['title'] ?? 'Product') . "</b>\n";
                $text .= __('bot.product_price_order', ['price' => $item->unit_price_usd], $lang) . "\n";
                $text .= __('bot.product_quantity', ['qty' => $item->qty], $lang) . "\n\n";

                // Show product details from snapshot
                if (!empty($snapshot['details'])) {
                    $text .= __('bot.product_details_order', [], $lang) . "\n";
                    $text .= "<code>" . $snapshot['details'] . "</code>\n\n";
                }

                $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
            }

            // Check if order can be reported (within 7 days and not already reported)
            $canReport = $order->created_at->diffInDays(now()) <= 7;
            $existingReport = \App\Models\ProductReport::where('order_id', $order->id)
                ->whereIn('status', ['pending', 'under_review'])
                ->exists();

            $buttons = [];

            if ($canReport && !$existingReport) {
                $buttons[] = [
                    ['text' => __('bot.report_issue', [], $lang), 'callback_data' => "report:{$order->id}"],
                ];
            }

            $buttons[] = [
                ['text' => __('bot.back_to_purchases', [], $lang), 'callback_data' => 'pur:0'],
            ];
            $buttons[] = [
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ];

            $keyboard = $telegram->createInlineKeyboard($buttons);
        }

        // Answer callback query
        $telegram->answerCallbackQuery($callback->id);

        // Send new message with order details
        $telegram->sendMessage(
            $callback->message->chat->id,
            $text,
            $keyboard
        );
    }

    private function handleReportSkip($callback): void
    {
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();
        if ($user) {
            app(ReportHandlers::class)->handleReportSkip($callback, $user);
        }
    }

    private function handleDepositStatus($callback, int $requestId): void
    {
        $telegram = app(\App\Services\TelegramService::class);
        $user = \App\Models\User::where('telegram_id', (string)$callback->from->id)->first();

        if (!$user) {
            return;
        }

        $lang = $user->language ?? 'ru';
        $depositRequest = \App\Models\DepositRequest::where('id', $requestId)
            ->where('user_id', $user->id)
            ->first();

        if (!$depositRequest) {
            $text = __('bot.product_not_found', [], $lang);
            $keyboard = $telegram->createInlineKeyboard([[
                ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
            ]]);
        } else {
            $deposit = \App\Models\Deposit::where('deposit_request_id', $depositRequest->id)
                ->where('status', 'confirmed')
                ->first();

            if ($deposit) {
                $text = __('bot.deposit_confirmed', [
                    'btc' => 0,
                    'usd' => $deposit->credited_usd,
                    'rate' => 0,
                    'balance' => $user->fresh()->balance_usd,
                ], $lang);

                $keyboard = $telegram->createInlineKeyboard([
                    [
                        ['text' => __('bot.my_balance', [], $lang), 'callback_data' => 'bal'],
                    ],
                    [
                        ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                    ],
                ]);
            } else {
                $text = __('bot.waiting_for_confirmation', [], $lang) . "\n\n";
                $text .= __('bot.deposit_address_label', ['address' => $depositRequest->address], $lang) . "\n\n";
                $text .= __('bot.status_label', ['status' => $depositRequest->status], $lang) . "\n\n";
                $text .= __('bot.waiting_for_transaction', [], $lang);

                $keyboard = $telegram->createInlineKeyboard([
                    [
                        ['text' => '🔄 ' . __('bot.check_status', [], $lang), 'callback_data' => "depstatus:{$requestId}"],
                    ],
                    [
                        ['text' => __('bot.home', [], $lang), 'callback_data' => 'menu'],
                    ],
                ]);
            }
        }

        // Answer callback query first
        $telegram->answerCallbackQuery($callback->id);

        $telegram->sendMessage(
            $callback->message->chat->id,
            $text,
            $keyboard
        );
    }
}
