<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProductReport extends Model
{
    protected $fillable = [
        'user_id',
        'order_id',
        'order_item_id',
        'product_id',
        'reason',
        'screenshot_url',
        'status',
        'admin_response',
        'admin_id',
        'refund_amount',
        'refunded_at',
    ];

    protected $casts = [
        'refunded_at' => 'datetime',
        'refund_amount' => 'decimal:2',
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    public function orderItem(): BelongsTo
    {
        return $this->belongsTo(OrderItem::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function admin(): BelongsTo
    {
        return $this->belongsTo(User::class, 'admin_id');
    }

    // Helper methods
    public function isValidScreenshotUrl(): bool
    {
        if (empty($this->screenshot_url)) {
            return true; // Optional field
        }

        $pattern = '/^(https?:\/\/)?(prnt\.sc|i\.imgur\.com)\/.+$/i';
        return preg_match($pattern, $this->screenshot_url) === 1;
    }

    public function canBeReported(): bool
    {
        // Can report within configured days of purchase
        $general = app(\App\Settings\GeneralSettings::class);
        return $this->order->created_at->diffInDays(now()) <= $general->max_report_days;
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'pending' => '⏳ Pending',
            'under_review' => '🔍 Under Review',
            'approved' => '✅ Approved',
            'denied' => '❌ Denied',
            'refunded' => '💰 Refunded',
            default => $this->status,
        };
    }
}
