<?php

namespace App\Services;

use neto737\BitGoSDK\BitGoSDK;
use neto737\BitGoSDK\Enum\CurrencyCode;
use Illuminate\Http\Request;

class BitGoService
{
    protected BitGoSDK $bitgo;
    protected string $walletId;
    protected bool $isTestnet;

    public function __construct()
    {
        $this->isTestnet = config('services.bitgo.testnet', true);
        
        $this->bitgo = new BitGoSDK(
            config('services.bitgo.api_key'),
            $this->isTestnet ? CurrencyCode::BITCOIN_TESTNET : CurrencyCode::BITCOIN,
            $this->isTestnet
        );
        
        $this->walletId = config('services.bitgo.wallet_id');
        $this->bitgo->walletId = $this->walletId;
    }

    public function createAddress(string $label, int $chain = 0): array
    {
        $response = $this->bitgo->createWalletAddress($chain, $label);

        if (!isset($response['address'])) {
            throw new \Exception('BitGo API did not return an address');
        }

        return [
            'address' => $response['address'],
            'id' => $response['id'] ?? null,
            'label' => $response['label'] ?? $label,
            'chain' => $response['chain'] ?? $chain,
            'wallet' => $response['wallet'] ?? $this->walletId,
        ];
    }

    public function verifyWebhook(Request $request): bool
    {
        $webhookSecret = config('services.bitgo.webhook_secret');
        
        if (empty($webhookSecret)) {
            return false;
        }

        $signature = $request->header('X-BitGo-Signature');
        
        if (empty($signature)) {
            return false;
        }

        $payload = $request->getContent();
        $expectedSignature = hash_hmac('sha256', $payload, $webhookSecret);
        
        return hash_equals($expectedSignature, $signature);
    }

    public function parseTx(Request $request): array
    {
        $data = $request->json()->all();
        
        $type = $data['type'] ?? null;
        
        if ($type === 'transfer') {
            return $this->parseTransferWebhook($data);
        } elseif ($type === 'transaction') {
            return $this->parseTransactionWebhook($data);
        }
        
        throw new \Exception('Unknown webhook type: ' . $type);
    }

    protected function parseTransferWebhook(array $data): array
    {
        $transfer = $data['transfer'] ?? [];
        
        return [
            'txid' => $transfer['txid'] ?? $transfer['id'] ?? null,
            'address' => $transfer['outputs'][0]['address'] ?? null,
            'value_sats' => $transfer['value'] ?? 0,
            'confirmations' => $transfer['confirmations'] ?? 0,
            'state' => $transfer['state'] ?? 'pending',
            'type' => $transfer['type'] ?? 'receive',
            'wallet_id' => $data['wallet'] ?? null,
            'raw_data' => $transfer,
        ];
    }

    protected function parseTransactionWebhook(array $data): array
    {
        $transaction = $data['transaction'] ?? [];
        
        return [
            'txid' => $transaction['id'] ?? $transaction['hash'] ?? null,
            'address' => $transaction['outputs'][0]['address'] ?? null,
            'value_sats' => $transaction['outputs'][0]['value'] ?? 0,
            'confirmations' => $transaction['confirmations'] ?? 0,
            'state' => $transaction['state'] ?? 'pending',
            'wallet_id' => $data['wallet'] ?? null,
            'raw_data' => $transaction,
        ];
    }

    public function getBalance(): array
    {
        $wallet = $this->bitgo->getWallet();
        
        return [
            'balance' => $wallet['balance'] ?? 0,
            'confirmed' => $wallet['confirmedBalance'] ?? 0,
            'spendable' => $wallet['spendableBalance'] ?? 0,
            'balance_string' => $wallet['balanceString'] ?? '0',
        ];
    }

    public function isTestnet(): bool
    {
        return $this->isTestnet;
    }

    public function getWalletId(): string
    {
        return $this->walletId;
    }

    public static function satoshisToBtc(int $satoshis): string
    {
        return bcdiv((string)$satoshis, '100000000', 8);
    }

    public static function btcToSatoshis(string $btc): int
    {
        return (int)bcmul($btc, '100000000', 0);
    }
}
