<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;

class RateService
{
    public function btcUsd(): string
    {
        return Cache::remember('btc_usd_rate', 300, function () {
            return $this->fetchBtcPrice();
        });
    }

    protected function fetchBtcPrice(): string
    {
        $providers = [
            'coingecko' => fn() => $this->fetchFromCoinGecko(),
            'coinbase' => fn() => $this->fetchFromCoinbase(),
            'blockchain' => fn() => $this->fetchFromBlockchain(),
        ];

        foreach ($providers as $name => $fetcher) {
            try {
                $price = $fetcher();
                if ($price && is_numeric($price) && $price > 0) {
                    return (string)$price;
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        throw new \Exception('Failed to fetch BTC price from all providers');
    }

    protected function fetchFromCoinGecko(): string
    {
        $response = Http::timeout(10)->get('https://api.coingecko.com/api/v3/simple/price', [
            'ids' => 'bitcoin',
            'vs_currencies' => 'usd',
        ]);

        if (!$response->successful()) {
            throw new \Exception('CoinGecko API failed');
        }

        $data = $response->json();
        
        if (!isset($data['bitcoin']['usd'])) {
            throw new \Exception('Invalid CoinGecko response');
        }

        return (string)$data['bitcoin']['usd'];
    }

    protected function fetchFromCoinbase(): string
    {
        $response = Http::timeout(10)->get('https://api.coinbase.com/v2/exchange-rates', [
            'currency' => 'BTC',
        ]);

        if (!$response->successful()) {
            throw new \Exception('Coinbase API failed');
        }

        $data = $response->json();
        
        if (!isset($data['data']['rates']['USD'])) {
            throw new \Exception('Invalid Coinbase response');
        }

        return (string)$data['data']['rates']['USD'];
    }

    protected function fetchFromBlockchain(): string
    {
        $response = Http::timeout(10)->get('https://blockchain.info/ticker');

        if (!$response->successful()) {
            throw new \Exception('Blockchain.info API failed');
        }

        $data = $response->json();
        
        if (!isset($data['USD']['last'])) {
            throw new \Exception('Invalid Blockchain.info response');
        }

        return (string)$data['USD']['last'];
    }

    public function clearCache(): void
    {
        Cache::forget('btc_usd_rate');
    }
}
