<?php

namespace App\Services;

use App\Models\User;
use App\Models\Deposit;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReferralService
{
    const COMMISSION_RATE = 0.10; // 10%

    /**
     * Process referral commission when a deposit is confirmed
     */
    public function processReferralCommission(Deposit $deposit): void
    {
        $user = $deposit->user;

        // Check if user was referred by someone
        if (!$user->referred_by) {
            return;
        }

        $referrer = User::find($user->referred_by);

        if (!$referrer) {
            return;
        }

        // Calculate commission (10% of deposit amount)
        $commission = $deposit->amount_usd * self::COMMISSION_RATE;

        DB::transaction(function () use ($referrer, $commission, $deposit) {
            // Add commission to referrer's balance
            $referrer->increment('balance_usd', $commission);
            
            // Track referral earnings
            $referrer->increment('referral_earnings', $commission);

            Log::info('Referral commission processed', [
                'referrer_id' => $referrer->id,
                'referred_user_id' => $deposit->user_id,
                'deposit_id' => $deposit->id,
                'commission' => $commission,
            ]);
        });
    }

    /**
     * Get referral statistics for a user
     */
    public function getReferralStats(User $user): array
    {
        $totalReferrals = $user->referrals()->count();
        $totalEarnings = $user->referral_earnings;
        
        // Get active referrals (users who have made at least one deposit)
        $activeReferrals = $user->referrals()
            ->whereHas('deposits', function ($query) {
                $query->where('status', 'confirmed');
            })
            ->count();

        return [
            'total_referrals' => $totalReferrals,
            'active_referrals' => $activeReferrals,
            'total_earnings' => $totalEarnings,
            'commission_rate' => self::COMMISSION_RATE * 100, // Convert to percentage
        ];
    }

    /**
     * Generate referral link for a user
     */
    public function generateReferralLink(User $user): string
    {
        $botUsername = config('services.telegram.bot_username');
        return "https://t.me/{$botUsername}?start=ref_{$user->referral_code}";
    }

    /**
     * Process referral code from /start command
     */
    public function processReferralCode(User $user, string $referralCode): bool
    {
        // Don't process if user already has a referrer
        if ($user->referred_by) {
            return false;
        }

        // Find referrer by code
        $referrer = User::where('referral_code', $referralCode)->first();

        if (!$referrer || $referrer->id === $user->id) {
            return false;
        }

        // Set referrer
        $user->referred_by = $referrer->id;
        $user->save();

        Log::info('Referral code processed', [
            'user_id' => $user->id,
            'referrer_id' => $referrer->id,
            'referral_code' => $referralCode,
        ]);

        return true;
    }
}
