<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class TelegramService
{
    protected string $botToken;
    protected string $apiUrl;

    public function __construct()
    {
        $this->botToken = config('services.telegram.bot_token');
        $this->apiUrl = "https://api.telegram.org/bot{$this->botToken}";
    }

    public function sendMessage(int $chatId, string $text, ?array $keyboard = null): array
    {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ];

        if ($keyboard) {
            $params['reply_markup'] = json_encode($keyboard);
        }

        $response = Http::post("{$this->apiUrl}/sendMessage", $params);

        return $response->json();
    }

    public function sendPhoto(int $chatId, string $photo, ?string $caption = null, ?array $keyboard = null): array
    {
        $params = [
            'chat_id' => $chatId,
            'photo' => $photo,
        ];

        if ($caption) {
            $params['caption'] = $caption;
            $params['parse_mode'] = 'HTML';
        }

        if ($keyboard) {
            $params['reply_markup'] = json_encode($keyboard);
        }

        $response = Http::post("{$this->apiUrl}/sendPhoto", $params);
        $result = $response->json();

        if (!($result['ok'] ?? false)) {
            \Log::error('TelegramService sendPhoto failed', [
                'error' => $result['description'] ?? 'Unknown error',
                'chat_id' => $chatId
            ]);
        }

        return $result;
    }

    public function sendPhotoFile(int $chatId, string $filePath, ?string $caption = null, ?array $keyboard = null): array
    {
        $params = [
            'chat_id' => $chatId,
        ];

        if ($caption) {
            $params['caption'] = $caption;
            $params['parse_mode'] = 'HTML';
        }

        if ($keyboard) {
            $params['reply_markup'] = json_encode($keyboard);
        }

        $response = Http::attach(
            'photo',
            file_get_contents($filePath),
            basename($filePath)
        )->post("{$this->apiUrl}/sendPhoto", $params);

        $result = $response->json();

        if (!($result['ok'] ?? false)) {
            \Log::error('TelegramService sendPhotoFile failed', [
                'error' => $result['description'] ?? 'Unknown error',
                'chat_id' => $chatId,
                'file' => basename($filePath)
            ]);
        }

        return $result;
    }

    public function answerCallbackQuery(string $callbackQueryId, ?string $text = null): array
    {
        $params = [
            'callback_query_id' => $callbackQueryId,
        ];

        if ($text) {
            $params['text'] = $text;
        }

        $response = Http::post("{$this->apiUrl}/answerCallbackQuery", $params);

        return $response->json();
    }

    public function editMessageText(int $chatId, int $messageId, string $text, ?array $keyboard = null): array
    {
        $params = [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'parse_mode' => 'HTML',
        ];

        if ($keyboard) {
            $params['reply_markup'] = json_encode($keyboard);
        }

        $response = Http::post("{$this->apiUrl}/editMessageText", $params);

        return $response->json();
    }

    public function sendChatAction(int $chatId, string $action = 'typing'): array
    {
        $params = [
            'chat_id' => $chatId,
            'action' => $action, // typing, upload_photo, record_video, upload_document, etc.
        ];

        $response = Http::post("{$this->apiUrl}/sendChatAction", $params);

        return $response->json();
    }

    public function createInlineKeyboard(array $buttons): array
    {
        return [
            'inline_keyboard' => $buttons,
        ];
    }

    public function setWebhook(string $url, ?string $secretToken = null): array
    {
        $params = [
            'url' => $url,
        ];

        if ($secretToken) {
            $params['secret_token'] = $secretToken;
        }

        $response = Http::post("{$this->apiUrl}/setWebhook", $params);

        return $response->json();
    }
}
