# 🐛 Bug Fix: TelegramUserData DTO Mapping

## ❌ **Issue**
```
Could not create `App\Bot\Data\TelegramUserData`: 
the constructor requires 4 parameters, 3 given.
```

## 🔍 **Root Cause**

The `TelegramUserData` DTO was not properly mapping Telegram's snake_case field names (`first_name`, `last_name`) to PHP properties.

When Telegram sends data like:
```json
{
  "id": 123456,
  "first_name": "John",
  "username": "johndoe"
}
```

Spatie's `laravel-data` couldn't automatically map `first_name` to the constructor parameter.

---

## ✅ **Solution**

### **1. Updated TelegramUserData.php**

**Before:**
```php
class TelegramUserData extends Data
{
    public function __construct(
        public int $id,
        public string $first_name,
        public ?string $username = null,
        public ?string $last_name = null,
    ) {}
}
```

**After:**
```php
use Spatie\LaravelData\Attributes\MapInputName;

class TelegramUserData extends Data
{
    public function __construct(
        public int $id,
        #[MapInputName('first_name')]
        public string $firstName,
        #[MapInputName('username')]
        public ?string $username = null,
        #[MapInputName('last_name')]
        public ?string $lastName = null,
    ) {}
}
```

### **2. Updated ProcessStartCommand.php**

**Before:**
```php
'name' => $message->from->first_name,
```

**After:**
```php
'name' => $message->from->firstName,
```

---

## 🎯 **How It Works**

The `#[MapInputName]` attribute tells Spatie's `laravel-data` to:
1. Look for `first_name` in the incoming data
2. Map it to the `$firstName` property
3. Same for `last_name` → `$lastName`

This allows us to:
- ✅ Use camelCase in PHP (PSR standards)
- ✅ Accept snake_case from Telegram API
- ✅ Automatic mapping via Spatie Data

---

## 🧪 **Testing**

Now when you send `/start` to the bot:

1. Telegram sends:
```json
{
  "message": {
    "from": {
      "id": 123456,
      "first_name": "John",
      "username": "johndoe"
    }
  }
}
```

2. `TelegramUserData::from()` correctly maps:
```php
TelegramUserData {
  id: 123456,
  firstName: "John",
  username: "johndoe",
  lastName: null
}
```

3. User is created with:
```php
User::create([
  'telegram_id' => '123456',
  'name' => 'John',  // from firstName
  'username' => 'johndoe'
])
```

---

## ✅ **Verification**

Test the fix:
```bash
# 1. Set webhook
php artisan telegram:set-webhook

# 2. Start queue worker
php artisan queue:work

# 3. Send /start to your bot
# Should work without errors now!
```

---

## 📝 **Files Changed**

1. ✅ `app/Bot/Data/TelegramUserData.php` - Added MapInputName attributes
2. ✅ `app/Actions/Telegram/ProcessStartCommand.php` - Updated to use firstName
3. ✅ `app/Console/Commands/SetTelegramWebhook.php` - Updated endpoint to /api/webhooks/telegram

---

## 🎊 **Status: FIXED**

The DTO mapping issue is now resolved and the bot should work correctly!

**Ready for testing!** 🚀
