# 🚀 ToolStore - Deployment Checklist

Use this checklist to deploy your ToolStore application to production.

---

## ✅ Pre-Deployment Checklist

### 1. Environment Configuration
- [ ] Copy `.env.example` to `.env`
- [ ] Set `APP_ENV=production`
- [ ] Set `APP_DEBUG=false`
- [ ] Generate `APP_KEY` with `php artisan key:generate`
- [ ] Configure database credentials
- [ ] Set up Redis connection

### 2. BitGo Configuration
- [ ] Create BitGo account (production or testnet)
- [ ] Create wallet and get `BITGO_WALLET_ID`
- [ ] Generate API key and set `BITGO_API_KEY`
- [ ] Generate webhook secret and set `BITGO_WEBHOOK_SECRET`
- [ ] Set `BITGO_TESTNET=false` for production
- [ ] Set `BITGO_REQUIRED_CONFIRMATIONS=2` (or your preference)

### 3. Telegram Bot Configuration
- [ ] Create bot with @BotFather
- [ ] Get bot token and set `TELEGRAM_BOT_TOKEN`
- [ ] Generate webhook secret and set `TELEGRAM_WEBHOOK_SECRET`
- [ ] Note your bot username

### 4. Database Setup
- [ ] Create production database
- [ ] Run `php artisan migrate`
- [ ] Run `php artisan db:seed` (creates admin user)
- [ ] Verify admin user created: `admin@toolstore.com`
- [ ] Change admin password in Filament panel

### 5. Redis Configuration
- [ ] Install Redis on server
- [ ] Set `QUEUE_CONNECTION=redis`
- [ ] Set `CACHE_DRIVER=redis`
- [ ] Test Redis connection

---

## 🌐 Webhook Setup

### Telegram Webhook

```bash
curl -X POST "https://api.telegram.org/bot<YOUR_BOT_TOKEN>/setWebhook" \
  -d "url=https://yourdomain.com/bot/webhook" \
  -d "secret_token=<YOUR_WEBHOOK_SECRET>"
```

**Verify webhook:**
```bash
curl "https://api.telegram.org/bot<YOUR_BOT_TOKEN>/getWebhookInfo"
```

### BitGo Webhook

1. Login to BitGo dashboard
2. Navigate to your wallet
3. Go to Webhooks section
4. Add new webhook:
   - **URL**: `https://yourdomain.com/webhooks/bitgo`
   - **Type**: Transfer (or Transaction)
   - **Confirmations**: 2 (or your preference)
5. Save webhook secret to `.env`

---

## 🔧 Server Configuration

### Required Software
- [ ] PHP 8.2 or higher
- [ ] MySQL 8.0 or higher
- [ ] Redis 7.x
- [ ] Composer 2.x
- [ ] Supervisor (for queue workers)
- [ ] Nginx or Apache with SSL

### PHP Extensions Required
- [ ] BCMath (for money calculations)
- [ ] cURL (for API calls)
- [ ] OpenSSL
- [ ] PDO MySQL
- [ ] Redis extension

### Supervisor Configuration

Create `/etc/supervisor/conf.d/toolstore-worker.conf`:

```ini
[program:toolstore-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/toolstore/artisan queue:work redis --sleep=3 --tries=3 --max-time=3600
autostart=true
autorestart=true
stopasgroup=true
killasgroup=true
user=www-data
numprocs=2
redirect_stderr=true
stdout_logfile=/path/to/toolstore/storage/logs/worker.log
stopwaitsecs=3600
```

**Start supervisor:**
```bash
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start toolstore-worker:*
```

### Horizon Configuration (Optional but Recommended)

Create `/etc/supervisor/conf.d/toolstore-horizon.conf`:

```ini
[program:toolstore-horizon]
process_name=%(program_name)s
command=php /path/to/toolstore/artisan horizon
autostart=true
autorestart=true
user=www-data
redirect_stderr=true
stdout_logfile=/path/to/toolstore/storage/logs/horizon.log
stopwaitsecs=3600
```

**Access Horizon dashboard:** `https://yourdomain.com/horizon`

---

## 🔐 Security Hardening

### File Permissions
```bash
cd /path/to/toolstore
sudo chown -R www-data:www-data storage bootstrap/cache
sudo chmod -R 775 storage bootstrap/cache
```

### Environment File
```bash
chmod 600 .env
```

### Disable Directory Listing
In your web server config, ensure directory listing is disabled.

### SSL Certificate
- [ ] Install SSL certificate (Let's Encrypt recommended)
- [ ] Force HTTPS in `.env`: `APP_URL=https://yourdomain.com`
- [ ] Update webhook URLs to use HTTPS

---

## 📊 Post-Deployment Verification

### 1. Test Admin Panel
- [ ] Visit `/admin`
- [ ] Login with `admin@toolstore.com`
- [ ] Change default password
- [ ] Create test category
- [ ] Create test product

### 2. Test Telegram Bot
- [ ] Send `/start` to bot
- [ ] Verify main menu appears
- [ ] Test `/balance` command
- [ ] Create deposit request
- [ ] Verify QR code displays

### 3. Test Deposit Flow (Testnet First!)
- [ ] Create deposit request in bot
- [ ] Send testnet BTC to address
- [ ] Monitor BitGo webhook logs
- [ ] Verify USD credited after confirmations
- [ ] Check Telegram notification received

### 4. Test Purchase Flow
- [ ] Manually credit test user balance (via admin)
- [ ] Browse products in bot
- [ ] Purchase a product
- [ ] Verify balance deducted
- [ ] Verify product details delivered
- [ ] Check "My Purchases" shows order

### 5. Monitor Logs
```bash
# Laravel logs
tail -f storage/logs/laravel.log

# Queue worker logs
tail -f storage/logs/worker.log

# Horizon logs (if using)
tail -f storage/logs/horizon.log
```

---

## 🎯 Production Optimization

### Cache Configuration
```bash
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### Optimize Autoloader
```bash
composer install --optimize-autoloader --no-dev
```

### Enable OPcache
Ensure OPcache is enabled in `php.ini`:
```ini
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=20000
```

---

## 📈 Monitoring Setup

### Application Monitoring
- [ ] Set up error tracking (Sentry, Bugsnag, etc.)
- [ ] Configure uptime monitoring
- [ ] Set up log aggregation

### Queue Monitoring
- [ ] Access Horizon dashboard: `/horizon`
- [ ] Monitor failed jobs
- [ ] Set up alerts for queue failures

### Database Monitoring
- [ ] Monitor database size
- [ ] Set up automated backups
- [ ] Test backup restoration

---

## 💾 Backup Strategy

### Database Backups
```bash
# Daily backup script
mysqldump -u username -p database_name > backup_$(date +%Y%m%d).sql
```

### File Backups
- [ ] Backup `.env` file (securely!)
- [ ] Backup `storage/` directory
- [ ] Backup database dumps

### Backup Schedule
- [ ] Daily automated backups
- [ ] Weekly full backups
- [ ] Monthly archive backups
- [ ] Test restoration quarterly

---

## 🚨 Emergency Procedures

### If Webhooks Stop Working

1. **Check webhook status:**
   ```bash
   curl "https://api.telegram.org/bot<TOKEN>/getWebhookInfo"
   ```

2. **Reset webhook:**
   ```bash
   curl -X POST "https://api.telegram.org/bot<TOKEN>/deleteWebhook"
   # Then set it again
   ```

3. **Check server logs:**
   ```bash
   tail -f storage/logs/laravel.log
   ```

### If Queue Workers Die

```bash
sudo supervisorctl restart toolstore-worker:*
# or
sudo supervisorctl restart toolstore-horizon
```

### If Double-Credits Occur

The system is designed to prevent this with idempotent checks, but if it happens:

1. Check `deposits` table for duplicate `credited_usd` entries
2. Review webhook logs
3. Manually adjust user balance if needed (via admin panel)
4. Investigate webhook signature verification

---

## 📝 Maintenance Tasks

### Daily
- [ ] Monitor error logs
- [ ] Check queue status
- [ ] Review failed jobs

### Weekly
- [ ] Review deposit transactions
- [ ] Check order completion rate
- [ ] Monitor BTC/USD rate accuracy
- [ ] Clear old cache if needed

### Monthly
- [ ] Review and optimize database
- [ ] Update dependencies: `composer update`
- [ ] Test backup restoration
- [ ] Review security logs

---

## ✅ Go-Live Checklist

Final checks before going live:

- [ ] All environment variables configured
- [ ] Database migrated and seeded
- [ ] Admin password changed from default
- [ ] Webhooks configured and tested
- [ ] SSL certificate installed
- [ ] Queue workers running
- [ ] Backups configured
- [ ] Monitoring set up
- [ ] Test deposit flow completed (testnet)
- [ ] Test purchase flow completed
- [ ] Error tracking configured
- [ ] Documentation reviewed

---

## 🎊 You're Live!

Once all checks pass:

1. Switch `BITGO_TESTNET=false` for production
2. Update BitGo webhook to production wallet
3. Announce bot to users
4. Monitor closely for first 24 hours
5. Celebrate! 🎉

---

## 📞 Support Resources

- **Laravel Docs**: https://laravel.com/docs
- **Filament Docs**: https://filamentphp.com/docs
- **BitGo API Docs**: https://www.bitgo.com/api
- **Telegram Bot API**: https://core.telegram.org/bots/api

---

**Remember**: Always test thoroughly on testnet before going to production!
