# Multi-Language & Referral System Guide

## 🌍 Multi-Language Support

Your Telegram bot now supports **English** and **Russian** languages with automatic user preference storage.

### Features
- ✅ **2 Languages**: English (en) and Russian (ru)
- ✅ **User Preference**: Language choice is saved per user
- ✅ **Easy Switching**: Users can change language anytime via bot menu
- ✅ **Comprehensive Translations**: All bot messages are translated

### How Users Change Language
1. Open bot menu with `/start`
2. Click **🌍 Language** button
3. Select preferred language (🇬🇧 English or 🇷🇺 Русский)
4. Language is saved automatically

### Configuration
Add your bot username to `.env`:
```env
TELEGRAM_BOT_USERNAME=your_bot_username
```

### Translation Service
Location: `app/Services/TranslationService.php`

**Usage in code:**
```php
$translation = app(TranslationService::class);
$text = $translation->get('welcome', $user->language, [
    'site_name' => 'My Store'
]);
```

### Available Translation Keys
- `welcome` - Welcome message
- `main_menu` - Main menu
- `browse_products` - Browse products button
- `my_orders` - My orders
- `my_balance` - Balance display
- `deposit` - Deposit button
- `help` - Help button
- `settings` - Settings
- `language` - Language selection
- `referral` - Referral program
- And many more...

---

## 👥 Referral System

Users earn **10% commission** on all deposits made by their referrals.

### Features
- ✅ **10% Commission**: Referrers earn 10% of referred user deposits
- ✅ **Unique Referral Codes**: Each user gets a unique 8-character code
- ✅ **Automatic Tracking**: System tracks referrals automatically
- ✅ **Real-time Earnings**: Commission credited immediately on deposit confirmation
- ✅ **Referral Statistics**: Users can view their referral stats

### How It Works

#### For Referrers:
1. User opens bot and goes to **👥 Referral Program**
2. Gets unique referral code and link
3. Shares link with friends
4. Earns 10% of every deposit their referrals make

#### For Referred Users:
1. Clicks referral link (e.g., `https://t.me/yourbot?start=ref_ABC12345`)
2. Starts bot - referral is automatically tracked
3. Makes deposits - referrer earns 10% commission

### Referral Link Format
```
https://t.me/{BOT_USERNAME}?start=ref_{REFERRAL_CODE}
```

Example: `https://t.me/myshopbot?start=ref_ABC12345`

### Database Structure

**New User Fields:**
- `language` - User's preferred language (en/ru)
- `referral_code` - Unique 8-character code
- `referred_by` - ID of user who referred them
- `referral_earnings` - Total earnings from referrals

### Commission Calculation

When a referred user makes a deposit:
```php
$commission = $deposit->amount_usd * 0.10; // 10%
```

**Example:**
- Referred user deposits $100
- Referrer earns $10 (added to balance)
- Referrer's `referral_earnings` increases by $10

### Referral Statistics

Users can view:
- Total referrals count
- Active referrals (those who made deposits)
- Total earnings from referrals
- Commission rate (10%)

### Services

#### ReferralService
Location: `app/Services/ReferralService.php`

**Methods:**
- `processReferralCommission(Deposit $deposit)` - Process commission on deposit
- `getReferralStats(User $user)` - Get user's referral statistics
- `generateReferralLink(User $user)` - Generate referral link
- `processReferralCode(User $user, string $code)` - Link user to referrer

### Bot Commands

**New Menu Items:**
- **👥 Referral Program** - View referral info and link
- **🌍 Language** - Change language preference

### Integration Points

#### 1. User Registration
When user starts bot with referral code:
```php
// In ProcessStartCommand
if (str_contains($message->text, 'ref_')) {
    preg_match('/ref_([A-Z0-9]+)/', $message->text, $matches);
    if (isset($matches[1])) {
        $this->referral->processReferralCode($user, $matches[1]);
    }
}
```

#### 2. Deposit Confirmation
When deposit is confirmed:
```php
// In BitGoWebhookController
$this->referral->processReferralCommission($deposit);
```

### Admin Features

**Filament Admin Panel:**
- View user referral codes
- See referral relationships
- Track referral earnings
- Monitor referral statistics

### Testing Referral System

1. **Create Test User A:**
   ```
   /start
   ```

2. **Get Referral Link:**
   - Click "👥 Referral Program"
   - Copy referral link

3. **Create Test User B:**
   - Open referral link in different Telegram account
   - Start bot

4. **Make Deposit as User B:**
   - Deposit BTC
   - Wait for confirmation

5. **Check User A Balance:**
   - Should see 10% commission added
   - Check "👥 Referral Program" for updated stats

### Monitoring

**Logs:**
```php
// Referral code processed
Log::info('Referral code processed', [
    'user_id' => $user->id,
    'referrer_id' => $referrer->id,
    'referral_code' => $referralCode,
]);

// Commission processed
Log::info('Referral commission processed', [
    'referrer_id' => $referrer->id,
    'referred_user_id' => $deposit->user_id,
    'deposit_id' => $deposit->id,
    'commission' => $commission,
]);
```

### Security

- ✅ Referral codes are unique and randomly generated
- ✅ Users cannot refer themselves
- ✅ Commission only paid on confirmed deposits
- ✅ Database transactions ensure consistency
- ✅ Comprehensive logging for audit trail

### Future Enhancements

Potential additions:
- Multi-tier referral system (referrals of referrals)
- Referral leaderboards
- Bonus rewards for top referrers
- Time-limited referral campaigns
- Referral analytics dashboard

---

## 📊 Statistics

### User Table Updates
```sql
ALTER TABLE users ADD COLUMN language VARCHAR(2) DEFAULT 'en';
ALTER TABLE users ADD COLUMN referral_code VARCHAR(20) UNIQUE;
ALTER TABLE users ADD COLUMN referred_by BIGINT UNSIGNED NULL;
ALTER TABLE users ADD COLUMN referral_earnings DECIMAL(12,2) DEFAULT 0;
```

### Migration
Run migration:
```bash
php artisan migrate
```

---

## 🚀 Quick Start

1. **Add bot username to .env:**
   ```env
   TELEGRAM_BOT_USERNAME=your_bot_username
   ```

2. **Run migration:**
   ```bash
   php artisan migrate
   ```

3. **Test language switching:**
   - Start bot: `/start`
   - Click: 🌍 Language
   - Select language

4. **Test referral system:**
   - Get referral link from bot
   - Share with test account
   - Make deposit
   - Check commission

---

## 📝 Notes

- Referral codes are generated automatically on user creation
- Language defaults to English (en) for new users
- Commission is paid instantly on deposit confirmation
- All translations are stored in `TranslationService.php`
- Referral earnings are tracked separately from regular balance

## 🎯 Summary

Your Telegram store bot now has:
- ✅ Multi-language support (English & Russian)
- ✅ 10% referral commission system
- ✅ Automatic referral tracking
- ✅ User language preferences
- ✅ Comprehensive translation service
- ✅ Real-time commission payments

Users can now use the bot in their preferred language and earn money by referring friends! 🎉
