# 🏗️ Spatie-Style Architecture Refactoring Plan

## 📋 **Current Violations & Required Changes**

Based on `.windsurf/rules.yml` Spatie-style principles:

### **Critical Violations:**
1. ❌ **Controller: 739 lines** → Must be ≤ 40 lines
2. ❌ **No DTOs** → Must use `spatie/laravel-data`
3. ❌ **No Actions** → Single-responsibility Action classes needed
4. ❌ **No Events** → Side effects should emit events
5. ❌ **No Jobs** → Heavy work must be queued
6. ❌ **Wrong namespace** → Must follow Domain-driven structure
7. ❌ **No strict types** → Missing `declare(strict_types=1)`
8. ❌ **Messages in controller** → Must centralize in `App\Bot\Messages`

---

## 🎯 **Target Architecture (Spatie-Style)**

### **Namespace Structure:**
```
app/
├── Domain/
│   ├── Users/
│   │   ├── Models/User.php
│   │   ├── Data/UserData.php
│   │   ├── Actions/CreateUserAction.php
│   │   └── Policies/UserPolicy.php
│   ├── Products/
│   │   ├── Models/Product.php
│   │   ├── Data/ProductData.php
│   │   └── Actions/PurchaseProductAction.php
│   └── Deposits/
│       ├── Models/Deposit.php
│       ├── Data/DepositData.php
│       └── Actions/CreditDepositAction.php
├── Bot/
│   ├── Handlers/
│   │   ├── StartCommandHandler.php
│   │   ├── BalanceCommandHandler.php
│   │   └── HelpCommandHandler.php
│   ├── Messages/
│   │   ├── WelcomeMessage.php
│   │   ├── BalanceMessage.php
│   │   └── HelpMessage.php
│   └── Data/
│       ├── TelegramUpdateData.php
│       ├── TelegramMessageData.php
│       └── TelegramCallbackData.php
├── Actions/
│   ├── Telegram/
│   │   ├── ProcessStartCommand.php
│   │   ├── ProcessBalanceCommand.php
│   │   ├── ProcessDepositRequest.php
│   │   └── ProcessPurchase.php
│   └── Deposits/
│       └── ProcessBitGoWebhook.php
├── Jobs/
│   ├── HandleTelegramUpdate.php
│   ├── ProcessBitGoWebhook.php
│   └── SendTelegramMessage.php
├── Events/
│   ├── UserRegistered.php
│   ├── DepositConfirmed.php
│   ├── ProductPurchased.php
│   └── BalanceUpdated.php
├── Listeners/
│   ├── SendWelcomeMessage.php
│   ├── NotifyDepositConfirmed.php
│   └── SendPurchaseReceipt.php
├── Services/
│   ├── TelegramService.php (thin wrapper)
│   ├── BitGoService.php (existing)
│   ├── RateService.php (existing)
│   └── QrService.php (existing)
└── Http/
    └── Controllers/
        └── Webhooks/
            ├── TelegramWebhookController.php (≤40 lines)
            └── BitGoWebhookController.php (≤40 lines)
```

---

## 📝 **Refactoring Steps**

### **Phase 1: Install Dependencies** ✅
- ✅ `spatie/laravel-data` - Installed
- ✅ `spatie/laravel-settings` - Installed
- ✅ `spatie/laravel-query-builder` - Installed

### **Phase 2: Create Directory Structure**
```bash
mkdir -p app/Domain/{Users,Products,Deposits}/{Models,Data,Actions,Policies}
mkdir -p app/Bot/{Handlers,Messages,Data}
mkdir -p app/Actions/Telegram
mkdir -p app/Jobs
mkdir -p app/Events
mkdir -p app/Listeners
mkdir -p app/Http/Controllers/Webhooks
```

### **Phase 3: Create DTOs (spatie/laravel-data)**

**TelegramUpdateData.php:**
```php
<?php

declare(strict_types=1);

namespace App\Bot\Data;

use Spatie\LaravelData\Data;

class TelegramUpdateData extends Data
{
    public function __construct(
        public int $updateId,
        public ?TelegramMessageData $message,
        public ?TelegramCallbackData $callbackQuery,
    ) {}
}
```

**TelegramMessageData.php:**
```php
<?php

declare(strict_types=1);

namespace App\Bot\Data;

use Spatie\LaravelData\Data;

class TelegramMessageData extends Data
{
    public function __construct(
        public int $messageId,
        public int $chatId,
        public int $fromId,
        public ?string $text,
        public ?string $username,
    ) {}
}
```

### **Phase 4: Create Action Classes**

**ProcessStartCommand.php:**
```php
<?php

declare(strict_types=1);

namespace App\Actions\Telegram;

use App\Bot\Data\TelegramMessageData;
use App\Domain\Users\Models\User;
use App\Events\UserRegistered;

class ProcessStartCommand
{
    public function execute(TelegramMessageData $message): void
    {
        $user = User::firstOrCreate(
            ['telegram_id' => (string)$message->fromId],
            [
                'name' => 'User',
                'username' => $message->username,
                'role' => 'user',
            ]
        );

        if ($user->wasRecentlyCreated) {
            event(new UserRegistered($user));
        }
    }
}
```

### **Phase 5: Create Events**

**UserRegistered.php:**
```php
<?php

declare(strict_types=1);

namespace App\Events;

use App\Domain\Users\Models\User;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class UserRegistered
{
    use Dispatchable, SerializesModels;

    public function __construct(
        public User $user
    ) {}
}
```

### **Phase 6: Create Jobs**

**HandleTelegramUpdate.php:**
```php
<?php

declare(strict_types=1);

namespace App\Jobs;

use App\Bot\Data\TelegramUpdateData;
use App\Actions\Telegram\ProcessStartCommand;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class HandleTelegramUpdate implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public TelegramUpdateData $update
    ) {}

    public function handle(): void
    {
        if ($this->update->message) {
            $this->handleMessage($this->update->message);
        }

        if ($this->update->callbackQuery) {
            $this->handleCallback($this->update->callbackQuery);
        }
    }

    private function handleMessage($message): void
    {
        if ($message->text === '/start') {
            app(ProcessStartCommand::class)->execute($message);
        }
    }

    private function handleCallback($callback): void
    {
        // Route to appropriate action
    }
}
```

### **Phase 7: Refactor Controllers (≤40 lines)**

**TelegramWebhookController.php:**
```php
<?php

declare(strict_types=1);

namespace App\Http\Controllers\Webhooks;

use App\Bot\Data\TelegramUpdateData;
use App\Http\Controllers\Controller;
use App\Http\Requests\TelegramWebhookRequest;
use App\Jobs\HandleTelegramUpdate;
use Illuminate\Http\JsonResponse;

class TelegramWebhookController extends Controller
{
    public function handle(TelegramWebhookRequest $request): JsonResponse
    {
        $updateData = TelegramUpdateData::from($request->validated());
        
        HandleTelegramUpdate::dispatch($updateData);
        
        return response()->json(['ok' => true]);
    }
}
```

### **Phase 8: Centralize Messages**

**WelcomeMessage.php:**
```php
<?php

declare(strict_types=1);

namespace App\Bot\Messages;

use App\Domain\Users\Models\User;

class WelcomeMessage
{
    public static function forNewUser(User $user): string
    {
        return "👋 <b>Welcome to ToolStore!</b>\n\n"
            . "Your trusted marketplace for digital products.\n\n"
            . "🎯 <b>What we offer:</b>\n"
            . "• Digital downloads & tools\n"
            . "• API credits & services\n"
            . "• Instant delivery\n"
            . "• Secure Bitcoin payments\n\n"
            . "Ready to start shopping?";
    }

    public static function forReturningUser(User $user, int $productCount): string
    {
        return "🏪 <b>Welcome back to ToolStore!</b>\n\n"
            . "Your balance: <b>\${$user->balance_usd}</b>\n"
            . "Available products: <b>{$productCount}</b>\n\n"
            . "What would you like to do?";
    }
}
```

---

## 🎯 **Benefits of This Architecture**

### **1. Small, Focused Classes**
- Controllers: ≤40 lines (just routing)
- Actions: Single responsibility
- Messages: Centralized, testable
- DTOs: Type-safe data transfer

### **2. Testability**
```php
// Easy to test Actions
it('processes start command', function () {
    $message = TelegramMessageData::from([...]);
    
    $action = new ProcessStartCommand();
    $action->execute($message);
    
    expect(User::count())->toBe(1);
});
```

### **3. Maintainability**
- Find code easily by domain
- Change one thing without breaking others
- Clear separation of concerns
- Easy to onboard new developers

### **4. Performance**
- Jobs queue heavy work
- Events decouple side effects
- Services cached where needed

---

## 📊 **Migration Strategy**

### **Step-by-Step:**
1. ✅ Install Spatie packages
2. ⏳ Create directory structure
3. ⏳ Create DTOs for Telegram data
4. ⏳ Extract one handler to Action (start with `/start`)
5. ⏳ Create corresponding Event & Listener
6. ⏳ Create Job for queued processing
7. ⏳ Refactor controller to dispatch Job
8. ⏳ Repeat for each handler
9. ⏳ Move messages to Message classes
10. ⏳ Add strict_types to all files
11. ⏳ Write Pest tests
12. ⏳ Configure PHPStan level max

---

## 🚀 **Next Immediate Steps**

1. Create directory structure
2. Create TelegramUpdateData DTO
3. Create ProcessStartCommand Action
4. Create UserRegistered Event
5. Create HandleTelegramUpdate Job
6. Refactor TelegramWebhookController to ≤40 lines

---

**This refactoring will transform the codebase to Spatie-quality standards!** 🎊
