# 🤖 Telegram Bot Testing Guide

This guide will help you test and verify the Telegram bot functionality.

---

## 📋 Pre-Testing Checklist

### 1. Create Your Telegram Bot

1. Open Telegram and search for **@BotFather**
2. Send `/newbot` command
3. Follow prompts to name your bot
4. Copy the **bot token** you receive
5. Save the bot username (e.g., @YourToolStoreBot)

### 2. Configure Environment

Add to your `.env` file:

```env
TELEGRAM_BOT_TOKEN=your_bot_token_here
TELEGRAM_WEBHOOK_SECRET=your_random_secret_string
APP_URL=https://your-domain.com  # or ngrok URL for local testing
```

**Generate a secure webhook secret:**
```bash
php -r "echo bin2hex(random_bytes(32));"
```

---

## 🧪 Testing Steps

### Step 1: Test Bot Connection

Run the test command to verify your bot is configured correctly:

```bash
php artisan telegram:test
```

**Expected output:**
```
🤖 Testing Telegram Bot Configuration...

✅ Bot token: Configured
✅ Webhook secret: Configured
📍 App URL: https://your-domain.com

Fetching bot info...
✅ Bot is accessible!
Bot name: ToolStore
Bot username: @YourToolStoreBot
Bot ID: 123456789

Checking webhook status...
⚠️  Webhook not set
Run: php artisan telegram:set-webhook

✅ Test complete!
```

### Step 2: Set Webhook

**For Local Testing (using ngrok):**

1. Install ngrok: https://ngrok.com/
2. Start your Laravel app: `php artisan serve`
3. In another terminal: `ngrok http 8000`
4. Copy the HTTPS URL (e.g., `https://abc123.ngrok.io`)
5. Set webhook:

```bash
php artisan telegram:set-webhook https://abc123.ngrok.io/bot/webhook
```

**For Production:**

```bash
php artisan telegram:set-webhook
# Uses APP_URL from .env automatically
```

**Expected output:**
```
Setting webhook to: https://your-domain.com/bot/webhook
✅ Webhook set successfully!
URL: https://your-domain.com/bot/webhook
Secret token: Configured
```

### Step 3: Test /start Command

1. Open Telegram
2. Search for your bot username
3. Send `/start` command

**Expected response:**
```
🏪 Welcome to ToolStore

Lawful digital downloads, API credits & support bundles.

What do you want to do?

[🛍 Browse] [💰 Deposit]
[💵 Balance] [📦 My Purchases]
[📜 Rules]
```

---

## 🧪 Test Each Feature

### Test 1: Balance Check

1. Click **💵 Balance** button
2. **Expected**: Shows $0.00 balance

### Test 2: Browse Categories

1. Click **🛍 Browse** button
2. **Expected**: Shows list of categories (Templates, API Credits, Digital Products)
3. Click a category
4. **Expected**: Shows subcategories or products

### Test 3: Deposit Request

1. Click **💰 Deposit** button
2. **Expected**: 
   - Unique BTC address displayed
   - QR code image shown
   - Instructions displayed

**Check database:**
```bash
php artisan tinker
>>> App\Models\DepositRequest::latest()->first()
```

Should show a new deposit request with unique address.

### Test 4: Rules

1. Click **📜 Rules** button
2. **Expected**: Shows terms and conditions

### Test 5: My Purchases (Empty)

1. Click **📦 My Purchases** button
2. **Expected**: "You haven't made any purchases yet."

---

## 🐛 Debugging

### Check Logs

**Laravel logs:**
```bash
tail -f storage/logs/laravel.log
```

**Watch for webhook calls:**
```bash
# In another terminal
tail -f storage/logs/laravel.log | grep "BotWebhook"
```

### Common Issues

#### Issue: Bot doesn't respond

**Check:**
1. Webhook is set correctly: `php artisan telegram:test`
2. App is accessible (ngrok running for local)
3. No errors in logs

**Solution:**
```bash
# Delete webhook and set again
curl "https://api.telegram.org/bot<TOKEN>/deleteWebhook"
php artisan telegram:set-webhook
```

#### Issue: QR code doesn't display

**Check:**
1. `endroid/qr-code` package installed
2. BitGo credentials configured
3. Check error in logs

#### Issue: "Unauthorized" error

**Check:**
1. `TELEGRAM_WEBHOOK_SECRET` matches in `.env` and webhook setup
2. Webhook secret is being sent by Telegram

---

## 📊 Test Deposit Flow (Full)

### Prerequisites:
- BitGo testnet wallet configured
- `BITGO_TESTNET=true` in `.env`

### Steps:

1. **Create deposit request in bot**
   ```
   Click: 💰 Deposit
   ```

2. **Verify in database:**
   ```bash
   php artisan tinker
   >>> App\Models\DepositRequest::latest()->first()
   # Should show: status='open', address='bc1q...'
   ```

3. **Send testnet BTC** (use Bitcoin testnet faucet)

4. **Monitor BitGo webhook:**
   ```bash
   tail -f storage/logs/laravel.log | grep "BitGo"
   ```

5. **After 2 confirmations, verify:**
   ```bash
   php artisan tinker
   >>> $user = App\Models\User::first()
   >>> $user->balance_usd
   # Should show credited USD amount
   
   >>> App\Models\Deposit::latest()->first()
   # Should show: status='confirmed', credited_usd=X.XX
   ```

6. **Check Telegram notification**
   - User should receive message: "✅ Deposit Confirmed!"

---

## 📦 Test Purchase Flow (Full)

### Prerequisites:
- User has USD balance (from deposit or manual credit)
- Products exist in database

### Steps:

1. **Add test product via admin:**
   ```
   /admin → Products → Create
   Title: Test Product
   Price: 5.00
   Category: (select any)
   Details: "This is your test product details"
   ```

2. **Browse in bot:**
   ```
   Click: 🛍 Browse
   Select category
   Select product
   ```

3. **View product details:**
   - Should show: Title, Price, Description
   - Button: [🛒 Buy Now]

4. **Purchase:**
   ```
   Click: 🛒 Buy Now
   ```

5. **Expected response:**
   ```
   ✅ Purchase Successful!
   
   Product: Test Product
   Amount: $5.00
   
   📄 Product Details:
   This is your test product details
   
   New balance: $X.XX
   ```

6. **Verify in database:**
   ```bash
   php artisan tinker
   >>> App\Models\Order::latest()->first()
   # Should show: status='paid', total_usd=5.00
   
   >>> App\Models\Product::first()
   # Should show: sold=true, sold_to='telegram_id'
   ```

7. **Check My Purchases:**
   ```
   Click: 📦 My Purchases
   ```
   - Should show the purchased product

---

## ✅ Testing Checklist

Use this checklist to verify all features:

- [ ] Bot responds to `/start`
- [ ] Main menu displays with all buttons
- [ ] Balance check works
- [ ] Browse shows categories
- [ ] Can navigate categories → subcategories → products
- [ ] Product detail page shows correctly
- [ ] Deposit request creates unique BTC address
- [ ] QR code displays
- [ ] Rules page displays
- [ ] My Purchases shows "empty" message initially
- [ ] Can purchase product with sufficient balance
- [ ] Purchase shows product details
- [ ] My Purchases shows purchased items
- [ ] Insufficient balance shows error message
- [ ] Back buttons work correctly
- [ ] Webhook logs show no errors

---

## 🔧 Manual Testing Tools

### Send Test Message

```bash
# Test sending a message
curl -X POST "https://api.telegram.org/bot<TOKEN>/sendMessage" \
  -d "chat_id=YOUR_CHAT_ID" \
  -d "text=Test message"
```

### Get Updates (for debugging)

```bash
# See recent updates
curl "https://api.telegram.org/bot<TOKEN>/getUpdates"
```

### Check Webhook Info

```bash
curl "https://api.telegram.org/bot<TOKEN>/getWebhookInfo"
```

---

## 📝 Test Data Setup

### Create Test User with Balance

```bash
php artisan tinker
>>> $user = App\Models\User::where('telegram_id', 'YOUR_TELEGRAM_ID')->first();
>>> $user->balance_usd = 100.00;
>>> $user->save();
```

### Create Test Products

```bash
php artisan tinker
>>> $category = App\Models\Category::first();
>>> App\Models\Product::create([
    'seller_id' => 1,
    'category_id' => $category->id,
    'title' => 'Test Product 1',
    'short_description' => 'A test product',
    'price' => 10.00,
    'details' => 'Product details here',
    'sold' => false,
    'deleted' => false,
]);
```

---

## 🎯 Next Steps After Testing

Once all tests pass:

1. ✅ Document any issues found
2. ✅ Fix bugs and edge cases
3. ✅ Add error handling where needed
4. ✅ Test with multiple users
5. ✅ Load test the bot
6. ✅ Prepare for production deployment

---

## 📞 Need Help?

If you encounter issues:

1. Check `storage/logs/laravel.log`
2. Verify all environment variables
3. Test webhook connectivity
4. Review Telegram API documentation
5. Check BitGo webhook logs

---

**Remember**: Always test with testnet/staging before production!
