# 🧪 Testing Guide - Spatie-Style Architecture

## ✅ **Implementation Status: 100% Complete**

All code is fully implemented with **ZERO TODOs**. Ready for testing!

---

## 📋 **Pre-Testing Checklist**

### **1. Environment Configuration**

Ensure your `.env` has:
```env
# Telegram
TELEGRAM_BOT_TOKEN=your_bot_token_here
TELEGRAM_WEBHOOK_SECRET=your_random_secret

# BitGo
BITGO_API_KEY=your_api_key
BITGO_WALLET_ID=your_wallet_id
BITGO_WEBHOOK_SECRET=your_webhook_secret
BITGO_TESTNET=true
BITGO_REQUIRED_CONFIRMATIONS=2

# Queue
QUEUE_CONNECTION=database  # or redis

# App
APP_URL=https://your-domain.com
```

### **2. Database Setup**

```bash
# Run migrations
php artisan migrate

# Seed database
php artisan db:seed
```

### **3. Queue Worker**

```bash
# Start queue worker (in separate terminal)
php artisan queue:work

# Or use Horizon (if configured)
php artisan horizon
```

---

## 🚀 **Testing Steps**

### **Step 1: Set Up Telegram Webhook**

#### **Option A: Using Artisan Command**
```bash
php artisan telegram:set-webhook https://your-domain.com/api/webhooks/telegram
```

#### **Option B: Using cURL**
```bash
curl -X POST "https://api.telegram.org/bot<YOUR_BOT_TOKEN>/setWebhook" \
  -d "url=https://your-domain.com/api/webhooks/telegram" \
  -d "secret_token=<YOUR_WEBHOOK_SECRET>"
```

#### **Verify Webhook**
```bash
curl "https://api.telegram.org/bot<YOUR_BOT_TOKEN>/getWebhookInfo"
```

Expected response:
```json
{
  "ok": true,
  "result": {
    "url": "https://your-domain.com/api/webhooks/telegram",
    "has_custom_certificate": false,
    "pending_update_count": 0
  }
}
```

---

### **Step 2: Test Commands**

Open your Telegram bot and test each command:

#### **Test /start Command**
```
Action: Send /start
Expected:
- Welcome message appears
- Inline keyboard with 6 buttons
- User created in database
- UserRegistered event fired
```

#### **Test /balance Command**
```
Action: Send /balance
Expected:
- Balance shows $0.00
- Statistics displayed
- Deposit and History buttons
```

#### **Test /help Command**
```
Action: Send /help
Expected:
- Help text with all commands
- How-to-use instructions
- Support contact info
```

#### **Test /history Command**
```
Action: Send /history
Expected:
- "No transactions yet" message
- Deposit button
```

---

### **Step 3: Test Callback Actions**

#### **Test Balance Callback**
```
Action: Click "💵 My Balance" button
Expected:
- Balance view appears
- Same as /balance command
- Message edited (not new message)
```

#### **Test Browse Callback**
```
Action: Click "🛍 Browse Products" button
Expected:
- Categories list appears
- Each category as button
- Back to menu button
```

#### **Test Deposit Callback**
```
Action: Click "💰 Deposit BTC" button
Expected:
- Unique BTC address generated
- QR code image sent
- Instructions displayed
- DepositRequest created in database
```

#### **Test Menu Callback**
```
Action: Click "🏠 Main Menu" button
Expected:
- Returns to main menu
- Welcome message shown
- Full keyboard displayed
```

---

### **Step 4: Test Purchase Flow**

#### **Setup: Add Test Balance**
```bash
php artisan tinker
>>> $user = App\Models\User::where('telegram_id', 'YOUR_TELEGRAM_ID')->first();
>>> $user->balance_usd = 100.00;
>>> $user->save();
```

#### **Test Purchase**
```
Action: 
1. Click "🛍 Browse Products"
2. Select a category
3. Select a product
4. Click "🛒 Buy Now"

Expected:
- Purchase confirmation shown
- Balance deducted
- Product details delivered
- ProductPurchased event fired
- Order created in database
```

---

### **Step 5: Test Queue Processing**

#### **Monitor Queue**
```bash
# Watch queue jobs
php artisan queue:work --verbose

# Or with Horizon
php artisan horizon
# Visit: http://your-domain.com/horizon
```

#### **Verify Jobs**
```
Expected:
- HandleTelegramUpdate jobs queued
- Jobs processed successfully
- No failed jobs
```

---

### **Step 6: Run Automated Tests**

#### **Run Pest Tests**
```bash
./vendor/bin/pest

# With coverage
./vendor/bin/pest --coverage

# Specific test
./vendor/bin/pest tests/Feature/Actions/ProcessStartCommandTest.php
```

Expected output:
```
✓ creates a new user when processing start command
✓ does not emit event for existing user
✓ shows balance for existing user
✓ does nothing for non-existent user
✓ routes start command to correct action
✓ handles message without text gracefully

Tests:  6 passed
Time:   0.5s
```

#### **Run PHPStan Analysis**
```bash
./vendor/bin/phpstan analyse

# With specific level
./vendor/bin/phpstan analyse --level=max
```

Expected:
```
[OK] No errors
```

---

## 🔍 **Debugging**

### **Check Logs**
```bash
# Laravel logs
tail -f storage/logs/laravel.log

# Queue logs
tail -f storage/logs/worker.log

# Filter for errors
tail -f storage/logs/laravel.log | grep ERROR
```

### **Check Database**
```bash
php artisan tinker

# Check users
>>> App\Models\User::count()

# Check deposit requests
>>> App\Models\DepositRequest::latest()->first()

# Check orders
>>> App\Models\Order::with('items')->latest()->first()
```

### **Test Webhook Locally (ngrok)**
```bash
# Install ngrok
brew install ngrok  # macOS
# or download from https://ngrok.com

# Start ngrok
ngrok http 8000

# Use ngrok URL for webhook
php artisan telegram:set-webhook https://abc123.ngrok.io/api/webhooks/telegram
```

---

## ✅ **Verification Checklist**

### **Architecture:**
- [ ] TelegramWebhookController receives updates
- [ ] HandleTelegramUpdate job queued
- [ ] Jobs processed by queue worker
- [ ] Actions executed correctly
- [ ] Messages sent via TelegramService
- [ ] Events fired and listeners called

### **Commands:**
- [ ] /start works
- [ ] /balance works
- [ ] /help works
- [ ] /history works

### **Callbacks:**
- [ ] Balance button works
- [ ] Deposit button works
- [ ] Browse button works
- [ ] Menu button works
- [ ] Buy button works

### **Database:**
- [ ] Users created
- [ ] DepositRequests created
- [ ] Orders created
- [ ] Products marked as sold

### **Events:**
- [ ] UserRegistered fires
- [ ] DepositConfirmed fires
- [ ] ProductPurchased fires

---

## 🐛 **Common Issues**

### **Issue: Webhook not receiving updates**
```bash
# Check webhook status
curl "https://api.telegram.org/bot<TOKEN>/getWebhookInfo"

# Delete and reset
curl -X POST "https://api.telegram.org/bot<TOKEN>/deleteWebhook"
php artisan telegram:set-webhook
```

### **Issue: Queue jobs not processing**
```bash
# Check queue connection
php artisan queue:work --verbose

# Clear failed jobs
php artisan queue:flush

# Restart queue
php artisan queue:restart
```

### **Issue: Database errors**
```bash
# Clear cache
php artisan cache:clear
php artisan config:clear

# Re-run migrations
php artisan migrate:fresh --seed
```

---

## 📊 **Success Criteria**

✅ All commands respond correctly
✅ All callbacks work
✅ Queue jobs process
✅ Database records created
✅ Events fire
✅ No errors in logs
✅ Pest tests pass
✅ PHPStan analysis passes

---

## 🎉 **When Testing is Complete**

You'll have verified:
- ✅ Complete Spatie-style architecture works
- ✅ All 9 Actions function correctly
- ✅ All 4 Messages display properly
- ✅ All 3 Events fire and Listeners execute
- ✅ Queue processing works
- ✅ Type safety maintained
- ✅ Zero errors

**Your Telegram bot is production-ready!** 🚀
