<?php

use App\Actions\Telegram\BrowseProductsAction;
use App\Bot\Data\TelegramCallbackData;
use App\Bot\Data\TelegramMessageData;
use App\Bot\Data\TelegramUserData;
use App\Bot\Data\TelegramChatData;
use App\Models\Category;
use App\Services\TelegramService;

uses()->group('telegram');

beforeEach(function () {
    $this->artisan('migrate:fresh');
    
    $this->telegram = Mockery::mock(TelegramService::class);
    $this->app->instance(TelegramService::class, $this->telegram);
    
    // Seed categories
    $this->artisan('db:seed', ['--class' => 'CategorySeeder']);
});

it('shows all active root categories', function () {
    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(
            id: 789,
            firstName: 'John',
            username: 'johndoe'
        ),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: 'browse'
    );

    $this->telegram->shouldReceive('createInlineKeyboard')
        ->once()
        ->with(Mockery::on(function ($buttons) {
            // Should have at least 3 category buttons + 1 main menu button
            return count($buttons) >= 4;
        }))
        ->andReturn([]);
        
    $this->telegram->shouldReceive('editMessageText')
        ->once()
        ->with(456, 123, Mockery::on(function ($text) {
            return str_contains($text, 'Browse Categories');
        }), Mockery::any());

    $action = new BrowseProductsAction($this->telegram);
    $action->execute($callback);
});

it('includes all seeded categories in buttons', function () {
    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: 'browse'
    );

    $categoryCount = Category::whereNull('parent_id')->where('is_active', true)->count();
    
    $this->telegram->shouldReceive('createInlineKeyboard')
        ->once()
        ->with(Mockery::on(function ($buttons) use ($categoryCount) {
            // Should have category buttons + main menu button
            return count($buttons) === $categoryCount + 1;
        }))
        ->andReturn([]);
        
    $this->telegram->shouldReceive('editMessageText')->once();

    $action = new BrowseProductsAction($this->telegram);
    $action->execute($callback);
    
    expect($categoryCount)->toBeGreaterThan(0);
});
