<?php

use App\Actions\Telegram\CreateDepositAction;
use App\Bot\Data\TelegramCallbackData;
use App\Bot\Data\TelegramMessageData;
use App\Bot\Data\TelegramUserData;
use App\Bot\Data\TelegramChatData;
use App\Models\User;
use App\Models\DepositRequest;
use App\Services\TelegramService;
use App\Services\BitGoService;
use App\Services\QrService;

uses()->group('telegram');

beforeEach(function () {
    $this->artisan('migrate:fresh');
    
    $this->telegram = Mockery::mock(TelegramService::class);
    $this->bitgo = Mockery::mock(BitGoService::class);
    $this->qr = Mockery::mock(QrService::class);
    
    $this->app->instance(TelegramService::class, $this->telegram);
    $this->app->instance(BitGoService::class, $this->bitgo);
    $this->app->instance(QrService::class, $this->qr);
});

it('creates deposit request with BTC address', function () {
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
    ]);

    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: 'dep:new'
    );

    // Mock BitGo response
    $this->bitgo->shouldReceive('createAddress')
        ->once()
        ->andReturn([
            'address' => 'tb1qtest123address',
            'label' => "user_{$user->id}_deposit_" . time(),
        ]);

    // Mock QR code generation
    $this->qr->shouldReceive('generateBitcoinQr')
        ->once()
        ->with('tb1qtest123address')
        ->andReturn('data:image/png;base64,test');

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('sendPhoto')
        ->once()
        ->with(456, 'data:image/png;base64,test', Mockery::on(function ($text) {
            return str_contains($text, 'Deposit Bitcoin') && 
                   str_contains($text, 'tb1qtest123address');
        }), Mockery::any());

    $action = new CreateDepositAction($this->telegram, $this->bitgo, $this->qr);
    $action->execute($callback);

    // Verify deposit request created
    expect(DepositRequest::where('user_id', $user->id)->exists())->toBeTrue();
    
    $depositRequest = DepositRequest::where('user_id', $user->id)->first();
    expect($depositRequest->address)->toBe('tb1qtest123address');
    expect($depositRequest->status)->toBe('open');
    expect($depositRequest->provider)->toBe('bitgo');
});

it('handles BitGo error gracefully', function () {
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
    ]);

    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: 'dep:new'
    );

    // Mock BitGo error
    $this->bitgo->shouldReceive('createAddress')
        ->once()
        ->andThrow(new Exception('BitGo API error'));

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('editMessageText')
        ->once()
        ->with(456, 123, Mockery::on(function ($text) {
            return str_contains($text, 'Failed to create deposit address');
        }), Mockery::any());

    $action = new CreateDepositAction($this->telegram, $this->bitgo, $this->qr);
    $action->execute($callback);

    // Verify no deposit request created
    expect(DepositRequest::where('user_id', $user->id)->exists())->toBeFalse();
});

it('does nothing for non-existent user', function () {
    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 999, firstName: 'Unknown', username: 'unknown'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 999, firstName: 'Unknown', username: 'unknown'),
            text: null
        ),
        data: 'dep:new'
    );

    $this->bitgo->shouldReceive('createAddress')->never();
    $this->qr->shouldReceive('generateBitcoinQr')->never();
    $this->telegram->shouldReceive('sendPhoto')->never();

    $action = new CreateDepositAction($this->telegram, $this->bitgo, $this->qr);
    $action->execute($callback);

    expect(true)->toBeTrue();
});
