<?php

use App\Actions\Telegram\ProcessHistoryCommand;
use App\Bot\Data\TelegramMessageData;
use App\Bot\Data\TelegramUserData;
use App\Bot\Data\TelegramChatData;
use App\Models\User;
use App\Models\Deposit;
use App\Models\Order;
use App\Models\Product;
use App\Models\Category;
use App\Services\TelegramService;

uses()->group('telegram');

beforeEach(function () {
    $this->artisan('migrate:fresh');
    
    $this->telegram = Mockery::mock(TelegramService::class);
    $this->app->instance(TelegramService::class, $this->telegram);
});

it('shows no transactions message for new user', function () {
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
    ]);

    $messageData = new TelegramMessageData(
        message_id: 123,
        chat: new TelegramChatData(id: 456, type: 'private'),
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        text: '/history'
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('sendMessage')
        ->once()
        ->with(456, Mockery::on(function ($text) {
            return str_contains($text, 'No transactions yet');
        }), Mockery::any());

    $action = new ProcessHistoryCommand($this->telegram);
    $action->execute($messageData);
});

it('shows recent deposits in history', function () {
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
    ]);

    // Create deposit request first
    $depositRequest = \App\Models\DepositRequest::create([
        'user_id' => $user->id,
        'provider' => 'bitgo',
        'chain' => 'btc',
        'address' => 'tb1qtest123',
        'label' => 'test_deposit',
        'status' => 'open',
    ]);

    // Create a confirmed deposit
    Deposit::create([
        'user_id' => $user->id,
        'deposit_request_id' => $depositRequest->id,
        'address' => 'tb1qtest123',
        'txid' => 'test_tx_' . uniqid(),
        'amount_satoshi' => 100000, // 0.001 BTC
        'credited_usd' => 50.00,
        'status' => 'confirmed',
        'credited_at' => now(),
    ]);

    $messageData = new TelegramMessageData(
        message_id: 123,
        chat: new TelegramChatData(id: 456, type: 'private'),
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        text: '/history'
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('sendMessage')
        ->once()
        ->with(456, Mockery::on(function ($text) {
            return str_contains($text, 'Recent Deposits') &&
                   str_contains($text, '$50');
        }), Mockery::any());

    $action = new ProcessHistoryCommand($this->telegram);
    $action->execute($messageData);
});

it('shows recent purchases in history', function () {
    $this->artisan('db:seed', ['--class' => 'CategorySeeder']);
    $this->artisan('db:seed', ['--class' => 'ProductSeeder']);
    
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
        'balance_usd' => 100.00,
    ]);

    $product = Product::where('sold', false)->first();
    
    // Create an order
    $order = Order::create([
        'user_id' => $user->id,
        'total_usd' => $product->price,
        'status' => 'paid',
    ]);

    $order->items()->create([
        'product_id' => $product->id,
        'qty' => 1,
        'unit_price_usd' => $product->price,
        'line_total_usd' => $product->price,
        'snapshot' => $product->toArray(),
    ]);

    $messageData = new TelegramMessageData(
        message_id: 123,
        chat: new TelegramChatData(id: 456, type: 'private'),
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        text: '/history'
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('sendMessage')
        ->once()
        ->with(456, Mockery::on(function ($text) use ($product) {
            return str_contains($text, 'Recent Purchases') &&
                   str_contains($text, '$' . $product->price);
        }), Mockery::any());

    $action = new ProcessHistoryCommand($this->telegram);
    $action->execute($messageData);
});

it('shows both deposits and purchases in history', function () {
    $this->artisan('db:seed', ['--class' => 'CategorySeeder']);
    $this->artisan('db:seed', ['--class' => 'ProductSeeder']);
    
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
        'balance_usd' => 100.00,
    ]);

    // Create deposit request first
    $depositRequest = \App\Models\DepositRequest::create([
        'user_id' => $user->id,
        'provider' => 'bitgo',
        'chain' => 'btc',
        'address' => 'tb1qtest456',
        'label' => 'test_deposit_2',
        'status' => 'open',
    ]);

    // Create deposit
    Deposit::create([
        'user_id' => $user->id,
        'deposit_request_id' => $depositRequest->id,
        'address' => 'tb1qtest456',
        'txid' => 'test_tx_' . uniqid(),
        'amount_satoshi' => 100000,
        'credited_usd' => 50.00,
        'status' => 'confirmed',
        'credited_at' => now(),
    ]);

    // Create order
    $product = Product::where('sold', false)->first();
    $order = Order::create([
        'user_id' => $user->id,
        'total_usd' => $product->price,
        'status' => 'paid',
    ]);

    $messageData = new TelegramMessageData(
        message_id: 123,
        chat: new TelegramChatData(id: 456, type: 'private'),
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        text: '/history'
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('sendMessage')
        ->once()
        ->with(456, Mockery::on(function ($text) {
            return str_contains($text, 'Recent Deposits') &&
                   str_contains($text, 'Recent Purchases');
        }), Mockery::any());

    $action = new ProcessHistoryCommand($this->telegram);
    $action->execute($messageData);
});

it('does nothing for non-existent user', function () {
    $messageData = new TelegramMessageData(
        message_id: 123,
        chat: new TelegramChatData(id: 456, type: 'private'),
        from: new TelegramUserData(id: 999, firstName: 'Unknown', username: 'unknown'),
        text: '/history'
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->never();
    $this->telegram->shouldReceive('sendMessage')->never();

    $action = new ProcessHistoryCommand($this->telegram);
    $action->execute($messageData);

    expect(true)->toBeTrue();
});
