<?php

use App\Actions\Telegram\PurchaseProductAction;
use App\Bot\Data\TelegramCallbackData;
use App\Bot\Data\TelegramMessageData;
use App\Bot\Data\TelegramUserData;
use App\Bot\Data\TelegramChatData;
use App\Events\ProductPurchased;
use App\Models\User;
use App\Models\Product;
use App\Models\Order;
use App\Services\TelegramService;
use Illuminate\Support\Facades\Event;

uses()->group('telegram');

beforeEach(function () {
    $this->artisan('migrate:fresh');
    
    $this->telegram = Mockery::mock(TelegramService::class);
    $this->app->instance(TelegramService::class, $this->telegram);
    
    // Seed products
    $this->artisan('db:seed', ['--class' => 'CategorySeeder']);
    $this->artisan('db:seed', ['--class' => 'ProductSeeder']);
});

it('successfully purchases product with sufficient balance', function () {
    Event::fake();
    
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
        'balance_usd' => 100.00,
    ]);

    $product = Product::where('sold', false)->first();
    expect($product)->not->toBeNull();

    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: "buy:{$product->id}"
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('editMessageText')
        ->once()
        ->with(456, 123, Mockery::on(function ($text) {
            return str_contains($text, 'Purchase Successful');
        }), Mockery::any());

    $action = new PurchaseProductAction($this->telegram);
    $action->execute($callback, $product->id);

    // Verify order created
    expect(Order::where('user_id', $user->id)->exists())->toBeTrue();
    
    // Verify product marked as sold
    expect($product->fresh()->sold)->toBeTrue();
    
    // Verify balance deducted
    $user->refresh();
    expect((float)$user->balance_usd)->toBe(100.00 - $product->price);
    
    // Verify event fired
    Event::assertDispatched(ProductPurchased::class);
});

it('rejects purchase with insufficient balance', function () {
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
        'balance_usd' => 5.00, // Not enough
    ]);

    $product = Product::where('sold', false)->first();

    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: "buy:{$product->id}"
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('editMessageText')
        ->once()
        ->with(456, 123, Mockery::on(function ($text) {
            return str_contains($text, 'Insufficient Balance');
        }), Mockery::any());

    $action = new PurchaseProductAction($this->telegram);
    $action->execute($callback, $product->id);

    // Verify no order created
    expect(Order::where('user_id', $user->id)->exists())->toBeFalse();
    
    // Verify product not sold
    expect($product->fresh()->sold)->toBeFalse();
});

it('rejects purchase of already sold product', function () {
    $user = User::create([
        'telegram_id' => '789',
        'name' => 'John',
        'username' => 'johndoe',
        'role' => 'user',
        'balance_usd' => 100.00,
    ]);

    $product = Product::where('sold', false)->first();
    $product->update(['sold' => true]);

    $callback = new TelegramCallbackData(
        id: 'callback123',
        from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
        message: new TelegramMessageData(
            message_id: 123,
            chat: new TelegramChatData(id: 456, type: 'private'),
            from: new TelegramUserData(id: 789, firstName: 'John', username: 'johndoe'),
            text: null
        ),
        data: "buy:{$product->id}"
    );

    $this->telegram->shouldReceive('createInlineKeyboard')->once()->andReturn([]);
    $this->telegram->shouldReceive('editMessageText')
        ->once()
        ->with(456, 123, Mockery::on(function ($text) {
            return str_contains($text, 'not available');
        }), Mockery::any());

    $action = new PurchaseProductAction($this->telegram);
    $action->execute($callback, $product->id);

    // Verify no new order created
    expect(Order::where('user_id', $user->id)->exists())->toBeFalse();
});
